/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "quakedef.h"
#include "r_local.h"

#define MAX_PARTICLES			4096	// default max # of particles at one
										//  time
#define ABSOLUTE_MIN_PARTICLES	512		// no fewer than this no matter what's
										//  on the command line

int		ramp1[8] = {0x6f, 0x6d, 0x6b, 0x69, 0x67, 0x65, 0x63, 0x61};
int		ramp2[8] = {0x6f, 0x6e, 0x6d, 0x6c, 0x6b, 0x6a, 0x68, 0x66};
int		ramp3[8] = {0x6d, 0x6b, 6, 5, 4, 3};

particle_t	*active_particles, *free_particles;

particle_t	*particles;
int			r_numparticles;

vec3_t			r_pright, r_pup, r_ppn;

/*
===============
R_InitParticles
===============
*/
void R_InitParticles (void)
{
	int		i;

	i = COM_CheckParm ("-particles");

	if (i)
	{
		r_numparticles = (int)(Q_atoi(com_argv[i+1]));
		if (r_numparticles < ABSOLUTE_MIN_PARTICLES)
			r_numparticles = ABSOLUTE_MIN_PARTICLES;
	}
	else
	{
		r_numparticles = MAX_PARTICLES;
	}

	particles = (particle_t *)
			Hunk_AllocName (r_numparticles * sizeof(particle_t), "particles");
}

void R_DarkFieldParticles (entity_t *ent)
{
	int			i, j, k;
	particle_t	*p;
	float		vel;
	vec3_t		dir;
	vec3_t		org;

	org[0] = ent->origin[0];
	org[1] = ent->origin[1];
	org[2] = ent->origin[2];
	for (i=-16 ; i<16 ; i+=8)
		for (j=-16 ; j<16 ; j+=8)
			for (k=0 ; k<32 ; k+=8)
			{
				if (!free_particles)
					return;
				p = free_particles;
				free_particles = p->next;
				p->next = active_particles;
				active_particles = p;
		
				p->die = cl.time + 0.2 + (rand()&7) * 0.02;
				p->born = cl.time;
				p->color = 150 + rand()%6;
				p->type = pt_slowgrav;
				
				dir[0] = j*8;
				dir[1] = i*8;
				dir[2] = k*8;
	
				p->org[0] = org[0] + i + (rand()&3);
				p->org[1] = org[1] + j + (rand()&3);
				p->org[2] = org[2] + k + (rand()&3);
	
				VectorNormalize (dir);						
				vel = 50 + (rand()&63);
				VectorScale (dir, vel, p->vel);
			}
}

/*
===============
R_EntityParticles
===============
*/

#define NUMVERTEXNORMALS	162
extern	float	r_avertexnormals[NUMVERTEXNORMALS][3];
vec3_t	avelocities[NUMVERTEXNORMALS];
float	beamlength = 16;
vec3_t	avelocity = {23, 7, 3};
float	partstep = 0.01;
float	timescale = 0.01;

void R_EntityParticles (entity_t *ent)
{
	int			count;
	int			i;
	particle_t	*p;
	float		angle;
	float		sr, sp, sy, cr, cp, cy;
	vec3_t		forward;
	float		dist;
	
	dist = 64;
	count = 50;

if (!avelocities[0][0])
{
for (i=0 ; i<NUMVERTEXNORMALS*3 ; i++)
avelocities[0][i] = (rand()&255) * 0.01;
}


	for (i=0 ; i<NUMVERTEXNORMALS ; i++)
	{
		angle = cl.time * avelocities[i][0];
		sy = sin(angle);
		cy = cos(angle);
		angle = cl.time * avelocities[i][1];
		sp = sin(angle);
		cp = cos(angle);
		angle = cl.time * avelocities[i][2];
		sr = sin(angle);
		cr = cos(angle);
	
		forward[0] = cp*cy;
		forward[1] = cp*sy;
		forward[2] = -sp;

		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 0.2;
		p->born = cl.time;
		p->color = 0x6f;
		p->type = pt_explode;
		p->fxtype = TE_BRIGHTFIELD;
		
		p->org[0] = ent->origin[0] + r_avertexnormals[i][0]*dist + forward[0]*beamlength;			
		p->org[1] = ent->origin[1] + r_avertexnormals[i][1]*dist + forward[1]*beamlength;			
		p->org[2] = ent->origin[2] + r_avertexnormals[i][2]*dist + forward[2]*beamlength;			
	}
}


/*
===============
R_ClearParticles
===============
*/
void R_ClearParticles (void)
{
	int		i;
	
	free_particles = &particles[0];
	active_particles = NULL;

	for (i=0 ;i<r_numparticles ; i++)
		particles[i].next = &particles[i+1];
	particles[r_numparticles-1].next = NULL;
}


void R_ReadPointFile_f (void)
{
	FILE	*f;
	vec3_t	org;
	int		r;
	int		c;
	particle_t	*p;
	char	name[MAX_OSPATH];
	
	sprintf (name,"maps/%s.pts", sv.name);

	COM_FOpenFile (name, &f);
	if (!f)
	{
		Con_Printf ("couldn't open %s\n", name);
		return;
	}
	
	Con_Printf ("Reading %s...\n", name);
	c = 0;
	for ( ;; )
	{
		r = fscanf (f,"%f %f %f\n", &org[0], &org[1], &org[2]);
		if (r != 3)
			break;
		c++;
		
		if (!free_particles)
		{
			Con_Printf ("Not enough free particles\n");
			break;
		}
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;
		
		p->die = 99999;
		p->born = cl.time;
		p->color = (-c)&15;
		p->type = pt_static;
		VectorCopy (vec3_origin, p->vel);
		VectorCopy (org, p->org);
	}

	fclose (f);
	Con_Printf ("%i points read\n", c);
}

/*
===============
R_ParseParticleEffect

Parse an effect out of the server message
===============
*/
void R_ParseParticleEffect (void)
{
	vec3_t		org, dir;
	int			i, count, msgcount, color;
	
	for (i=0 ; i<3 ; i++)
		org[i] = MSG_ReadCoord ();
	for (i=0 ; i<3 ; i++)
		dir[i] = MSG_ReadChar () * (1.0/16);
	msgcount = MSG_ReadByte ();
	color = MSG_ReadByte ();

if (msgcount == 255)
	count = 1024;
else
	count = msgcount;
	
	R_RunParticleEffect (org, dir, color, count, TE_FREE);
}
	
/*
===============
R_ParticleExplosion

===============
*/
void R_ParticleExplosion (vec3_t org)
{
	int			i, j, num;
	particle_t	*p;

#ifdef GLQUAKE		
	num = 128;
#else
	num = 512;
#endif

	num = num * r_particledensity.value;
	for (i=0 ; i<num ; i++)
	{
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->fxtype = TE_EXPLOSION;
		p->die = cl.time + 0.5;
		p->born = cl.time;
		p->color = ramp1[0];
		p->ramp = rand()&3;
		if (i & 1)
		{
			p->type = pt_explode;
			for (j=0 ; j<3 ; j++)
			{
				p->org[j] = org[j] + ((rand()%32)-16);
				p->vel[j] = (rand()%1024)-512;
			}
		}
		else
		{
			p->type = pt_explode2;
			for (j=0 ; j<3 ; j++)
			{
				p->org[j] = org[j] + ((rand()%32)-16);
				p->vel[j] = (rand()%1024)-512;
			}
		}
	}
}

/*
===============
R_ParticleExplosion2

===============
*/
void R_ParticleExplosion2 (vec3_t org, int colorStart, int colorLength)
{
	int			i, j, num;
	particle_t	*p;
	int			colorMod = 0;

#ifdef GLQUAKE		
	num = 128;
#else
	num = 512;
#endif

	num = num * r_particledensity.value;
	for (i=0; i<num; i++)
	{
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 0.3;
		p->born = cl.time;
		p->color = colorStart + (colorMod % colorLength);
		colorMod++;

		p->type = pt_blob;
		p->fxtype = TE_EXPLOSION2;
		for (j=0 ; j<3 ; j++)
		{
			p->org[j] = org[j] + ((rand()%32)-16);
			p->vel[j] = (rand()%512)-256;
		}
	}
}

/*
===============
R_BlobExplosion

===============
*/
void R_BlobExplosion (vec3_t org)
{
	int			i, j, num;
	particle_t	*p;

#ifdef GLQUAKE		
	num = 128;
#else
	num = 512;
#endif

	num = num * r_particledensity.value;	
	for (i=0 ; i<num ; i++)
	{
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;
		p->die = cl.time + 1 + (rand()&8)*0.05;
		p->born = cl.time;
		p->fxtype = TE_TAREXPLOSION;

		if (i & 1)
		{
			p->type = pt_blob;
			p->color = 66 + rand()%6;
			for (j=0 ; j<3 ; j++)
			{
				p->org[j] = org[j] + ((rand()%32)-16);
				p->vel[j] = (rand()%512)-256;
			}
		}
		else
		{
			p->type = pt_blob2;
			p->color = 150 + rand()%6;
			for (j=0 ; j<3 ; j++)
			{
				p->org[j] = org[j] + ((rand()%32)-16);
				p->vel[j] = (rand()%512)-256;
			}
		}
	}
}

/*
===============
R_RunParticleEffect

===============
*/
void R_RunParticleEffect (vec3_t org, vec3_t dir, int color, int count, int te_type)
{
	int			i, j;
	particle_t	*p;

#ifdef GLQUAKE
	count = count * 0.25;
	if (count < 1)
		count = 1;
#endif

	count = count * r_particledensity.value;
	for (i=0 ; i<count ; i++)
	{
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;
		p->fxtype = te_type;
		p->die = cl.time + 0.1*(rand()%5);
		p->color = (color&~7) + (rand()&7);
		p->type = pt_slowgrav;
		for (j=0 ; j<3 ; j++)
		{
			p->org[j] = org[j] + ((rand()&15)-8);
			p->vel[j] = dir[j]*25 + (rand()%100)-50;
		}
	}
}


/*
===============
R_LavaSplash

===============
*/
void R_LavaSplash (vec3_t org)
{
	int			i, j, k, count;
	particle_t	*p;
	float		vel;
	vec3_t		dir;

#ifdef GLQUAKE
	count = 4;
#else
	count = 16;
#endif
	count = count * r_particledensity.value;

	for (i=-count ; i<count ; i++)
		for (j=-count ; j<count ; j++)
			for (k=0 ; k<1 ; k++)
			{
				if (!free_particles)
					return;
				p = free_particles;
				free_particles = p->next;
				p->next = active_particles;
				active_particles = p;
		
				p->die = cl.time + 1;
				p->born = cl.time;
				p->color = 224 + (rand()&7);
				p->type = pt_slowgrav;
				p->fxtype = TE_LAVASPLASH;
				
				dir[0] = j*32 + (rand()&7);
				dir[1] = i*32 + (rand()&7);
				dir[2] = (rand()&20);
	
				p->org[0] = org[0] + dir[0];
				p->org[1] = org[1] + dir[1];
				p->org[2] = org[2] + (rand()&63);
	
				VectorNormalize (dir);						
				vel = 150 + (rand()&63);
				VectorScale (dir, vel, p->vel);
			}

	for (i=-count ; i<count ; i++)
		for (j=-count ; j<count ; j++)
			for (k=0 ; k<1 ; k++)
			{
				if (!free_particles)
					return;
				p = free_particles;
				free_particles = p->next;
				p->next = active_particles;
				active_particles = p;
		
				p->die = cl.time + 1;
				p->born = cl.time;
				p->color = 224 + (rand()&7);
				p->type = pt_slowgrav;
				p->fxtype = TE_LAVASPLASH;
				
				dir[0] = j*8 + (rand()&7);
				dir[1] = i*8 + (rand()&7);
				dir[2] = 512;
	
				p->org[0] = org[0] + dir[0];
				p->org[1] = org[1] + dir[1];
				p->org[2] = org[2] + (rand()&63);
	
				VectorNormalize (dir);						
				vel = 150 + (rand()&63);
				VectorScale (dir, vel, p->vel);
			}
}

/*
===============
R_TeleportSplash

===============
*/
void R_TeleportSplash (vec3_t org)
{
	int			i, j, k, num;
	particle_t	*p;
	float		vel;
	vec3_t		dir;

#ifdef GLQUAKE
	num = 8;
#else
	num = 4;
#endif
	
	for (i=-16 ; i<16 ; i+=num)
		for (j=-16 ; j<16 ; j+=num)
			for (k=-24 ; k<32 ; k+=num)
			{
				if (!free_particles)
					return;
				p = free_particles;
				free_particles = p->next;
				p->next = active_particles;
				active_particles = p;
		
				p->die = cl.time + 0.2 + (rand()&7) * 0.02;
				p->born = cl.time;
				p->color = 7 + (rand()&7);
				p->type = pt_slowgrav;
				p->fxtype = TE_TELEPORT;
				
				dir[0] = j*8;
				dir[1] = i*8;
				dir[2] = k*8;
	
				p->org[0] = org[0] + i + (rand()&3);
				p->org[1] = org[1] + j + (rand()&3);
				p->org[2] = org[2] + k + (rand()&3);
	
				VectorNormalize (dir);						
				vel = 50 + (rand()&63);
				VectorScale (dir, vel, p->vel);
			}
}

void R_RocketTrail (vec3_t start, vec3_t end, int type)
{
	vec3_t		vec;
	float		len;
	int			j;
	particle_t	*p;
	int			dec;
	static int	tracercount;

	VectorSubtract (end, start, vec);
	len = VectorNormalize (vec);
	if (type < 128)
#ifdef GLQUAKE
		dec = 20/r_particledensity.value;
#else
		dec = 10/r_particledensity.value;
#endif
	else
	{
		dec = 1;
		type -= 128;
	}
	while (len > 0)
	{
		len -= dec;

		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;
		
		VectorCopy (vec3_origin, p->vel);

		switch (type)
		{
			case 0:	// rocket trail
				p->fxtype = TE_THRUST;
				p->die = cl.time + 0.25*r_particleduration.value;
				p->born = cl.time;
				p->ramp = (rand()&3);
				p->color = ramp3[(int)p->ramp];
				p->type = pt_fire;
				for (j=0 ; j<3 ; j++)
					p->org[j] = start[j] + ((rand()%6)-3);
				break;

			case 1:	// smoke 
				p->die = cl.time + 3*r_particleduration.value;
				p->born = cl.time;
				p->fxtype = TE_TRAIL;
				p->ramp = (rand()&3) + 2;
				p->color = ramp3[(int)p->ramp];
				p->type = pt_fire;
				for (j=0 ; j<3 ; j++)
				{
					p->org[j] = start[j];
					p->vel[j] = (rand()%20)-10;
				}
				break;

			case 2:	// blood
				p->fxtype = TE_BLOODTRAIL;
				p->die = cl.time + 2;
				p->born = cl.time;
				p->type = pt_grav;
				p->color = 67 + (rand()&3);
				for (j=0 ; j<3 ; j++)
				{
					p->vel[j] = (rand()%10)-5;
					p->org[j] = start[j] + ((rand()%6)-3);
				}
				len -= (50/r_particledensity.value);
				break;

			case 3:

			case 5:	// tracer
				p->type = pt_static;
				if (type == 3 && fh1)
				{
					p->die = cl.time + 0.5 * r_particleduration.value;
					p->born = cl.time;
					p->fxtype = TE_WIZSPIKE;
					p->color = 230 + ((tracercount&4)<<1);
					for (j=0; j<3 ; j++)
						p->vel[j] = (end[j] - start[j]) * 20;
					VectorCopy (start, p->org);
				}
				else if (type == 3 && !fh1)
				{
					p->die = cl.time + 1 * r_particleduration.value;
					p->born = cl.time;
					p->fxtype = TE_WIZSPIKE;
					p->color = 230 + ((tracercount&4)<<1);
					VectorCopy (start, p->org);
				}
				else if (type != 3 && fh1)
				{
					p->die = cl.time + 0.5 * r_particleduration.value;
					p->born = cl.time;
					p->fxtype = TE_KNIGHTSPIKE;
					p->color = 230 + ((tracercount&4)<<1);
					VectorCopy (start, p->org);
					for (j=0 ; j<3 ; j++)
						p->org[j] = start[j] + ((rand()&20)-10);
				}
				else if (type != 3 && !fh1)
				{
					p->die = cl.time + 1 * r_particleduration.value;
					p->born = cl.time;
					p->fxtype = TE_KNIGHTSPIKE;
					p->color = 230 + ((tracercount&4)<<1);
					VectorCopy (start, p->org);
				}
				tracercount++;
				break;

			case 4:	// slight blood
				p->fxtype = TE_BLOODTRAIL;
				p->die = cl.time + 2;
				p->born = cl.time;
				p->type = pt_grav;
				p->color = 67 + (rand()&3);
				for (j=0 ; j<3 ; j++)
				{
					p->org[j] = start[j] + ((rand()%6)-3);
					p->vel[j] = (rand()%10)-5;
				}
				len -= (50/r_particledensity.value);
				break;

			case 6:	// voor trail
				p->fxtype = TE_SPARK;
				p->die = cl.time + 2*r_particleduration.value;
				p->born = cl.time;
				p->color = 9*16 + 8 + (rand()&3);
				p->type = pt_static;
				p->die = cl.time + 0.3;
				for (j=0 ; j<3 ; j++)
					p->org[j] = start[j] + ((rand()&15)-8);
				break;
		}
		VectorAdd (start, vec, start);
	}
}

extern	cvar_t	sv_gravity;

#ifndef GLQUAKE
/*
===============
R_DrawParticles
===============
*/

void R_DrawParticles (void)
{
	particle_t		*p, *kill;
	float			grav;
	int				i;
	float			time2, time3;
	float			time1;
	float			dvel;
	float			frametime;
	
	D_StartParticles ();

	VectorScale (vright, xscaleshrink, r_pright);
	VectorScale (vup, yscaleshrink, r_pup);
	VectorCopy (vpn, r_ppn);
	frametime = cl.time - cl.oldtime;
	time3 = frametime * 15;
	time2 = frametime * 10; // 15;
	time1 = frametime * 5;
	grav = frametime * sv_gravity.value * 0.05;
	dvel = 4*frametime;
	
	for ( ;; ) 
	{
		kill = active_particles;
		if (kill && kill->die < cl.time)
		{
			active_particles = kill->next;
			kill->next = free_particles;
			free_particles = kill;
			continue;
		}
		break;
	}

	for (p=active_particles ; p ; p=p->next)
	{
		for ( ;; )
		{
			kill = p->next;
			if (kill && kill->die < cl.time)
			{
				p->next = kill->next;
				kill->next = free_particles;
				free_particles = kill;
				continue;
			}
			break;
		}

		D_DrawParticle (p);
		p->org[0] += p->vel[0]*frametime;
		p->org[1] += p->vel[1]*frametime;
		p->org[2] += p->vel[2]*frametime;
		
		switch (p->type)
		{
		case pt_static:
			break;
		case pt_fire:
			p->ramp += time1;
			if (p->ramp >= 6)
				p->ramp = 6;
			else
				p->color = ramp3[(int)p->ramp];
			//p->vel[2] += grav;
			break;

		case pt_explode:
			p->ramp += time2;
			if (p->ramp >=8)
				p->die = -1;
			else
				p->color = ramp1[(int)p->ramp];
			for (i=0 ; i<3 ; i++)
				p->vel[i] += p->vel[i]*dvel;
			p->vel[2] -= grav;
			break;

		case pt_explode2:
			p->ramp += time3;
			if (p->ramp >=8)
				p->die = -1;
			else
				p->color = ramp2[(int)p->ramp];
			for (i=0 ; i<3 ; i++)
				p->vel[i] -= p->vel[i]*frametime;
			p->vel[2] -= grav;
			break;

		case pt_blob:
			for (i=0 ; i<3 ; i++)
				p->vel[i] += p->vel[i]*dvel;
			p->vel[2] -= grav;
			break;

		case pt_blob2:
			for (i=0 ; i<2 ; i++)
				p->vel[i] -= p->vel[i]*dvel;
			p->vel[2] -= grav;
			break;

		case pt_slowgrav:
			p->vel[2] -= grav;
			break;
		}
	}
	D_EndParticles ();
}
#endif

/*
===============
R_DrawUltraParticles
===============
*/
void R_DrawUltraParticles (void)
{
	particle_t		*p, *kill;
	float			grav;
	int				i, j, bob;
	float			time2, time3;
	float			time1;
	float			dvel;
	float			frametime;
	float			ttlive;
	float			alpha;
	float			age;
	
	vec3_t			up, right;
	float			scale;

	//vec3_t	org;
	vec3_t	v;
	float		radius;
	float		distance;
	float		intensity;
	
#ifdef GLQUAKE
	if (gl_fogenable.value)
		glDisable(GL_FOG);
	glDepthMask (0);
	glDisable (GL_TEXTURE_2D);
	glShadeModel (GL_SMOOTH);
	glEnable (GL_BLEND);
	glBlendFunc (GL_ONE, GL_ONE);

	glPushMatrix();			
	glBegin(GL_TRIANGLE_FAN);

	VectorScale (vup, 1, up);
	VectorScale (vright, 1, right);

	frametime = cl.time - cl.oldtime;
	time3 = frametime * 15;
	time2 = frametime * 10; // 15;
	time1 = frametime * 5;
	grav = frametime * sv_gravity.value * 0.05;
	dvel = 4*frametime;
	
	for ( ;; ) 
	{
		kill = active_particles;
		if (kill && kill->die < cl.time)
		{
			active_particles = kill->next;
			kill->next = free_particles;
			free_particles = kill;
			continue;
		}
		break;
	}


	for (p=active_particles ; p ; p=p->next)
	{
		for ( ;; )
		{
			kill = p->next;
			if (kill && kill->die < cl.time)
			{
				p->next = kill->next;
				kill->next = free_particles;
				free_particles = kill;
				continue;
			}
			break;
		}

		// time to live per particle
		ttlive = p->die - cl.time;
		age = cl.time - p->born;

		// set radius and intensity per particle based on
		// time to live
		bob = p->fxtype;
		switch(bob)
		{
			case TE_WIZSPIKE:
				if (fh1)
				{
					radius = 10 + (0.5 - ttlive) * 100;
					intensity = 1 - (age*2);
					glColor3f(0.8f*intensity, 0.4f*intensity, 0.1);
				}
				else
				{
					radius = 5 + ttlive * 10;
					intensity = ttlive;
					glColor3f(0.4f*intensity, 0.7f*intensity, 0.1);
				}
				break;

			case TE_KNIGHTSPIKE:
				if (fh1)
				{
					radius = 10 + ttlive * 40;
					intensity = 1 - (age * 2);
					glColor3f(0.8f*intensity, 0.4f*intensity, 0.1);
				}
				else
				{
					radius = 5 + ttlive * 10;
					intensity = ttlive;
					glColor3f(0.8f*intensity, 0.4f*intensity, 0.1);
				}
				break;

			case TE_SPARK:
				radius = 20;
				intensity = ttlive*4;
				glColor3f(0.1f*intensity, 0.4f*intensity, 0.8f*intensity);
				break;

			case TE_BLOODTRAIL:
				radius = 3 + (2 - ttlive) * 3;
				intensity = ttlive;
				glColor3f( 0.23 * intensity, 0.02 * intensity , 0.02 * intensity);
				break;

			case TE_TRAIL:
				radius = 10 + (3*r_particleduration.value - ttlive) * 10;
				intensity = ttlive * 0.1;
				glColor3f(0.4f*intensity, 0.4f*intensity, 0.5f*intensity);
				break;

			case TE_THRUST:
				radius = 20;
				intensity = 1 - (age*4);
				glColor3f(0.8f*intensity, 0.4f*intensity, 0.1f*intensity);
				break;

			case TE_FREE:
				radius = 2;
				intensity = ttlive * 0.5;
				glColor3ubv ((byte *)&d_8to24table[(int)p->color]);
				break;

			case TE_GUNSHOT:
				radius = 8;
				intensity = ttlive * 2;
				glColor3f(0.25f*intensity, 0.25f*intensity, 0.25f*intensity);
				break;

			case TE_SUPERSPIKE:
				radius = 5;
				intensity = ttlive * 10;
				glColor3f(0.3f*intensity, 0.3f*intensity, 0.3f*intensity);
				break;

			case TE_EXPLOSION:
				radius = 30 + ttlive * 20;
				intensity = ttlive * 2;
				glColor3f(0.8f*intensity, 0.4f*intensity, 0.1);
				break;

			case TE_TAREXPLOSION:
				radius = 30 - (ttlive * 6);
				intensity = ttlive * 0.2;
				glColor3f(0.8f*intensity, 0.1f*intensity, 0.6f*intensity);
				break;

			case TE_LAVASPLASH:
				radius = 30 + ((1 - ttlive) * 40);
				intensity = 1 - age;
				glColor3f(0.8f*intensity, 0.4f*intensity, 0.1f);
				p->vel[2] -= grav * 8;
				break;

			case TE_TELEPORT:
				radius = 3 - (ttlive * 0.1);
				intensity = ttlive * 3;
				glColor3f(1.0f*intensity, 1.0f*intensity, 1.0f*intensity);
				break;

			case TE_EXPLOSION2:
				radius = 30 - (ttlive * 6);
				intensity = 1 - (age * 0.2); //ttlive * 0.2;
				glColor3f(0.8f*intensity, 0.4f*intensity, 0.1f);
				break;

			case TE_BRIGHTFIELD:
				radius =  10;
				intensity = ttlive * 5;
				glColor3f(0.9f*intensity, 0.9f*intensity, 0.9f*intensity);
				break;
		}

		if (radius <= 0.0f) radius = 1.0f;
		
		VectorSubtract(p->org, r_origin, v);
		distance = Length(v);
		if ( (distance > (radius * 4)) || (p->fxtype == TE_WIZSPIKE || p->fxtype == TE_SPARK) )
		{
			glBegin(GL_TRIANGLE_FAN);
			for (i=0 ; i<3 ; i++)
				v[i] = p->org[i] - vpn[i] * radius;
			glVertex3fv(v);
			glColor3f(0.0f, 0.0f, 0.0f);
			for (i=16; i>=0; i--)
			{
				float a = i/16.0f * M_PI * 2;
				for (j=0; j<3; j++)
				 v[j] =	p->org[j] + 
					vright[j]*cos(a)*radius +
					vup[j]*sin(a)*radius;
				glVertex3fv(v);
			}
			p->org[0] += p->vel[0]*frametime;
			p->org[1] += p->vel[1]*frametime;
			p->org[2] += p->vel[2]*frametime;
			glEnd();
		}
	}

	glEnd();
	glPopMatrix();
	glColor3f (1,1,1);
	glDisable (GL_BLEND);
	glEnable (GL_TEXTURE_2D);
	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glDepthMask (1);
	if (gl_fogenable.value)
		glEnable(GL_FOG);
#endif
}
