/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "quakedef.h"
#include "r_local.h"

#define MAX_PARTICLES			4096	// default max # of particles at one
										//  time
#define ABSOLUTE_MIN_PARTICLES	512		// no fewer than this no matter what's
										//  on the command line

int		ramp1[8] = {0x6f, 0x6d, 0x6b, 0x69, 0x67, 0x65, 0x63, 0x61};
int		ramp2[8] = {0x6f, 0x6e, 0x6d, 0x6c, 0x6b, 0x6a, 0x68, 0x66};
int		ramp3[8] = {0x6d, 0x6b, 6, 5, 4, 3};

particle_t	*active_particles, *free_particles;

particle_t	*particles;
int			r_numparticles;
vec3_t			r_pright, r_pup, r_ppn;

int		smoke_tex, fire_tex, radar_tex, explo_tex, debris_tex, spark_tex, blood_tex;

extern float over_dim;
float	rotater[4];

/*=============
R_InitParticles
=============*/
void R_InitParticles (void)
{
	int		i;

	i = COM_CheckParm ("-particles");

	if (i)
	{
		r_numparticles = (int)(Q_atoi(com_argv[i+1]));
		if (r_numparticles < ABSOLUTE_MIN_PARTICLES)
			r_numparticles = ABSOLUTE_MIN_PARTICLES;
	}
	else
	{
		r_numparticles = MAX_PARTICLES;
	}

	particles = (particle_t *)
			Hunk_AllocName (r_numparticles * sizeof(particle_t), "particles");

	// load textures
#ifdef GLQUAKE
	// setup rotation
	for (i=4; i>=0; i--)
		rotater[i] = i/4.0f * M_PI * 2;
	
	smoke_tex = TA_LoadTGA("particles/smoke1");
	i = TA_LoadTGA("particles/smoke2");
	i = TA_LoadTGA("particles/smoke3");
	i = TA_LoadTGA("particles/smoke4");

	fire_tex = TA_LoadTGA("particles/fire1");
	i = TA_LoadTGA("particles/fire2");
	i = TA_LoadTGA("particles/fire3");
	i = TA_LoadTGA("particles/fire4");
	i = TA_LoadTGA("particles/fire5");
	i = TA_LoadTGA("particles/fire6");

	radar_tex = TA_LoadTGA("particles/radar");

	explo_tex = TA_LoadTGA("particles/explo1");
	i = TA_LoadTGA("particles/explo2");
	i = TA_LoadTGA("particles/explo3");
	i = TA_LoadTGA("particles/explo4");

	debris_tex = TA_LoadTGA("particles/debris1");
	i = TA_LoadTGA("particles/debris2");
	i = TA_LoadTGA("particles/debris3");
	i = TA_LoadTGA("particles/debris4");

	blood_tex = TA_LoadTGA("particles/splat1");
	i = TA_LoadTGA("particles/splat2");
	i = TA_LoadTGA("particles/splat3");
	i = TA_LoadTGA("particles/splat4");
	
	spark_tex = TA_LoadTGA("particles/spark1");

#endif
}
/*=============================================================
R_CreateParticle - silly little function that makes a particle.
=============================================================*/
void R_CreateParticle(int count,
					  int fxtype,
					  int drawtype,
					  int tex_num,
					  float color,
					  ptype_t type,
					  float born,
					  float die,
					  float angle,
					  float v_angle,
					  float alpha,
					  float scale,
					  vec3_t org,
					  int org_rand,
					  int vel,
					  int vel_z,
					  int power)
{
	int i, j;
	particle_t *p;

	for(i=0; i<count;i++)
	{
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;
		
		p->fxtype = fxtype;
		p->drawtype = drawtype;
		p->tex_num = tex_num;
		p->color = color;
		p->type = type;
		p->born = born;
		p->die = die;
		p->angle = angle;
		p->v_angle = v_angle;
		p->alpha = alpha;
		p->scale = scale;

		if (power == 0) power = 1;

		//origin
		if (org_rand == 0)
		{
			VectorCopy(org, p->org);
		}
		else
		{
			for (j=0 ; j<3 ; j++)
				p->org[j] = org[j] + (rand()%(org_rand*2))-org_rand;
		}

		// velocity
		if (vel_z != 0 && vel != 0)
		{
			for (j=0 ; j<2 ; j++)
				p->vel[j] = ((rand()%(vel*2))-vel) * power;
			p->vel[2] = (rand()%vel_z) + power;
		}
		else if (vel != 0)
		{
			for (j=0 ; j<3 ; j++)
				p->vel[j] = ((rand()%(vel*2))-vel) * power;
		}
		else if (vel_z != 0)
		{
			p->vel[2] = (rand()%vel_z) + power;
			p->vel[0] = 0.0f;
			p->vel[1] = 0.0f;
		}
		else
		{
			p->vel[0] = 0.0f;
			p->vel[1] = 0.0f;
			p->vel[2] = 0.0f;
		}

	}
}

void R_DarkFieldParticles (entity_t *ent)
{
	int			i, j, k;
	particle_t	*p;
	float		vel;
	vec3_t		dir;
	vec3_t		org;

	org[0] = ent->origin[0];
	org[1] = ent->origin[1];
	org[2] = ent->origin[2];
	for (i=-16 ; i<16 ; i+=8)
		for (j=-16 ; j<16 ; j+=8)
			for (k=0 ; k<32 ; k+=8)
			{
				if (!free_particles)
					return;
				p = free_particles;
				free_particles = p->next;
				p->next = active_particles;
				active_particles = p;
		
				p->die = cl.time + 0.2 + (rand()&7) * 0.02;
				p->born = cl.time;
				p->color = 150 + rand()%6;
				p->type = pt_slowgrav;
				
				dir[0] = j*8;
				dir[1] = i*8;
				dir[2] = k*8;
	
				p->org[0] = org[0] + (i + (rand()&3) * worldscale.value);
				p->org[1] = org[1] + (j + (rand()&3) * worldscale.value);
				p->org[2] = org[2] + (k + (rand()&3) * worldscale.value);
	
				VectorNormalize (dir);						
				vel = 50 + (rand()&63);
				VectorScale (dir, vel, p->vel);
			}
}

/*===============
R_EntityParticles
===============*/

#define NUMVERTEXNORMALS	162
extern	float	r_avertexnormals[NUMVERTEXNORMALS][3];
vec3_t	avelocities[NUMVERTEXNORMALS];
float	beamlength = 16;
vec3_t	avelocity = {23, 7, 3};
float	partstep = 0.01;
float	timescale = 0.01;

void R_EntityParticles (entity_t *ent)
{
	int			count;
	int			i;
	particle_t	*p;
	float		angle;
	float		sr, sp, sy, cr, cp, cy;
	vec3_t		forward;
	float		dist;
	
	dist = 64;
	count = 50;

if (!avelocities[0][0])
{
for (i=0 ; i<NUMVERTEXNORMALS*3 ; i++)
avelocities[0][i] = (rand()&255) * 0.01;
}


	for (i=0 ; i<NUMVERTEXNORMALS ; i++)
	{
		angle = cl.time * avelocities[i][0];
		sy = sin(angle);
		cy = cos(angle);
		angle = cl.time * avelocities[i][1];
		sp = sin(angle);
		cp = cos(angle);
		angle = cl.time * avelocities[i][2];
		sr = sin(angle);
		cr = cos(angle);
	
		forward[0] = cp*cy;
		forward[1] = cp*sy;
		forward[2] = -sp;

		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 0.2;
		p->born = cl.time;
		p->color = 0x6f;
		p->type = pt_explode;
		p->fxtype = TE_BRIGHTFIELD;
		p->drawtype = PFX_GLOW;
		
		p->org[0] = ent->origin[0] + r_avertexnormals[i][0]*dist + forward[0]*beamlength;
		p->org[1] = ent->origin[1] + r_avertexnormals[i][1]*dist + forward[1]*beamlength;
		p->org[2] = ent->origin[2] + r_avertexnormals[i][2]*dist + forward[2]*beamlength;
	}
}

/*=================================
		R_ModelParticles
=================================*/
void R_ModelParticles (int type, vec3_t org)
{
	int			i, j, num, fxtype;
	particle_t	*p;

	if (cl.paused)
		return;
	
	fxtype = 0;
	if (type == 1)
		fxtype = TE_SMALLGLOW;
	else if (type == 2)
		fxtype = TE_BIGGLOW;
	else if (type == 3)
		fxtype = TE_BIGGLOW;
	else if (type == 4)
		fxtype = TE_TORCHGLOW;
	else if (type == 9)
		fxtype = TE_BOLTGLOW;
	else if (type == 11)
		fxtype = TE_GOLDGLOW;
	else if (type == 12)
		fxtype = TE_BLUEGLOW;
	
	if (!free_particles || fxtype == 0)
		return;

	p = free_particles;
	free_particles = p->next;
	p->next = active_particles;
	active_particles = p;
	
	p->die = cl.time + 0.01;
	p->born = cl.time;
	p->color = ramp1[0];
	p->ramp = rand()&3;
	p->type = pt_static;
	p->fxtype = fxtype;
	p->drawtype = PFX_GLOW;
	for (j=0 ; j<3 ; j++)
		p->org[j] = org[j];
}

/*==============
R_ClearParticles
==============*/
void R_ClearParticles (void)
{
	int		i;
	
	free_particles = &particles[0];
	active_particles = NULL;

	for (i=0 ;i<r_numparticles ; i++)
		particles[i].next = &particles[i+1];
	particles[r_numparticles-1].next = NULL;
}

void R_ReadPointFile_f (void)
{
	FILE	*f;
	vec3_t	org;
	int		r;
	int		c;
	particle_t	*p;
	char	name[MAX_OSPATH];
	
	sprintf (name,"maps/%s.pts", sv.name);

	COM_FOpenFile (name, &f);
	if (!f)
	{
		Con_Printf ("couldn't open %s\n", name);
		return;
	}
	
	Con_Printf ("Reading %s...\n", name);
	c = 0;
	for ( ;; )
	{
		r = fscanf (f,"%f %f %f\n", &org[0], &org[1], &org[2]);
		if (r != 3)
			break;
		c++;
		
		if (!free_particles)
		{
			Con_Printf ("Not enough free particles\n");
			break;
		}
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;
		
		p->die = 99999;
		p->born = cl.time;
		p->color = (-c)&15;
		p->type = pt_static;
		VectorCopy (vec3_origin, p->vel);
		VectorCopy (org, p->org);
	}

	fclose (f);
	Con_Printf ("%i points read\n", c);
}

/*===============================================================
R_ParseParticleEffect - Parse an effect out of the server message
===============================================================*/
void R_ParseParticleEffect (void)
{
	vec3_t		org, dir;
	int			i, count, msgcount, color;
	
	for (i=0 ; i<3 ; i++)
		org[i] = MSG_ReadCoord ();
	for (i=0 ; i<3 ; i++)
		dir[i] = MSG_ReadChar () * (1.0/16);
	msgcount = MSG_ReadByte ();
	color = MSG_ReadByte ();

	if (msgcount == 255)
		count = 1024;
	else
		count = msgcount;

	if (color > 63 && color < 80 )
		R_RunParticleEffect (org, dir, color, count, TE_BLOOD);
	else
		R_RunParticleEffect (org, dir, color, count, TE_FREE);
}
	
/*=================
R_ParticleExplosion
=================*/
void R_ParticleExplosion (vec3_t org)
{
	int			i, j, k, count;
	particle_t	*p;
	vec3_t		dir;
	float		vel;

#ifdef GLQUAKE		
	count = 2;
#else
	count = 8;
#endif

	count = count * r_particledensity.value;

	for (k=0 ; k<count ; k++)
	{
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 1;
		p->born = cl.time;
		p->color = ramp1[0];
		p->fxtype = TE_EXPLOSION;
		p->drawtype = PFX_GLOW;
		p->type = pt_explode;

		for (j=0 ; j<3 ; j++)
		{
			p->org[j] = org[j] + (((rand()%32)-16) * worldscale.value);
			p->vel[j] = (rand()%128)-64;
		}
	}

	for (i=-count ; i<count ; i++)
		for (j=-count ; j<count ; j++)
			for (k=0 ; k<1 ; k++)
			{
				if (!free_particles)
					return;
				p = free_particles;
				free_particles = p->next;
				p->next = active_particles;
				active_particles = p;

				p->die = cl.time + 2 + (rand()&2);
				p->born = cl.time;
				p->color = ramp1[0];
				p->type = pt_slowgrav;
				p->fxtype = TE_EXPLOSION;
				p->drawtype = PFX_SPRITE;
				p->tex_num = explo_tex;
				p->angle = rand()%7;
				p->v_angle = (rand()%40)-20;

				dir[0] = j*32 + (rand()&31);
				dir[1] = i*32 + (rand()&31);
				dir[2] = 512;
	
				p->org[0] = org[0] + (rand()&7);
				p->org[1] = org[1] + (rand()&7);
				p->org[2] = org[2] + (rand()&7);
	
				VectorNormalize (dir);						
				vel = 300 + (rand()&127);
				VectorScale (dir, vel * worldscale.value, p->vel);
				if (i & 1)
					p->type = pt_explode;
				else
					p->type = pt_explode2;
			}

	for (i=-count ; i<count ; i++)
		for (j=-count ; j<count ; j++)
			for (k=0 ; k<1 ; k++)
			{
				if (!free_particles)
					return;
				p = free_particles;
				free_particles = p->next;
				p->next = active_particles;
				active_particles = p;
		
				p->die = cl.time + 2;
				p->born = cl.time;
				p->color = ramp1[0];
				p->type = pt_slowgrav;
				p->fxtype = TE_EXPLOSION;
				p->drawtype = PFX_SPRITE;
				p->tex_num = explo_tex;
				p->angle = rand()%7;
				p->v_angle = (rand()%40)-20;
				
				dir[0] = j*32 + (rand()&15);
				dir[1] = i*32 + (rand()&15);
				dir[2] = (rand()&20);
	
				p->org[0] = org[0] + (rand()&7);
				p->org[1] = org[1] + (rand()&7);
				p->org[2] = org[2] + (rand()&7);
	
				VectorNormalize (dir);						
				vel = 300 + (rand()&127);
				VectorScale (dir, vel * worldscale.value, p->vel);

				if (i & 1)
					p->type = pt_explode;
				else
					p->type = pt_explode2;
			}
}

/*==================
R_ParticleExplosion2
==================*/
void R_ParticleExplosion2 (vec3_t org, int colorStart, int colorLength)
{
	int			i, j, num;
	particle_t	*p;
	int			colorMod = 0;

#ifdef GLQUAKE		
	num = 128;
#else
	num = 512;
#endif

	num = num * r_particledensity.value;
	for (i=0; i<num; i++)
	{
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 0.3;
		p->born = cl.time;
		p->color = colorStart + (colorMod % colorLength);
		colorMod++;

		p->type = pt_blob;
		p->fxtype = TE_EXPLOSION2;
		p->drawtype = PFX_GLOW;
		for (j=0 ; j<3 ; j++)
		{
			p->org[j] = org[j] + (((rand()%32)-16) * worldscale.value);
			p->vel[j] = (rand()%512)-256;
		}
	}
}

/*===============
R_BlobExplosion
===============*/
void R_BlobExplosion (vec3_t org)
{
	int			i, j, num;
	particle_t	*p;

#ifdef GLQUAKE		
	num = 128;
#else
	num = 512;
#endif

	num = num * r_particledensity.value;	
	for (i=0 ; i<num ; i++)
	{
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;
		p->die = cl.time + 1 + (rand()&8)*0.05;
		p->born = cl.time;
		p->fxtype = TE_TAREXPLOSION;
		p->drawtype = PFX_GLOW;

		if (i & 1)
		{
			p->type = pt_blob;
			p->color = 66 + rand()%6;
			for (j=0 ; j<3 ; j++)
			{
				p->org[j] = org[j] + (((rand()%32)-16) * worldscale.value);
				p->vel[j] = (rand()%512)-256;
			}
		}
		else
		{
			p->type = pt_blob2;
			p->color = 150 + rand()%6;
			for (j=0 ; j<3 ; j++)
			{
				p->org[j] = org[j] + (((rand()%32)-16) * worldscale.value);
				p->vel[j] = (rand()%512)-256;
			}
		}
	}
}

/*=================
R_RunParticleEffect
=================*/
void R_RunParticleEffect (vec3_t org, vec3_t dir, int color, int count, int type)
{
	count = count * r_particledensity.value;
	if (type == TE_SUPERSPIKE)
		R_CreateParticle(count, TE_SUPERSPIKE, PFX_DART, 0, (color&~7) + (rand()&7), pt_slowgrav, cl.time, cl.time + 0.25, 0, 0, 1, 1, org, 0, 511, 255, 0);
	else if (type == TE_SPIKE)
		R_CreateParticle(count, TE_SPIKE, PFX_DART, 0, (color&~7) + (rand()&7), pt_slowgrav, cl.time, cl.time + 0.25, 0, 0, 1, 1, org, 0, 255, 127, 0);
	else if (type == TE_GUNSHOT)
	{
		R_CreateParticle(count, TE_GUNSHOT, PFX_SPRITE, debris_tex + (rand()%4), 95 + (rand()&15), pt_slowgrav, cl.time,
				cl.time + 1, rand()%7, (rand()%127 - 63), 1, 2 + (rand()%2), org, 0, 128, 200, 0);
		R_CreateParticle(2, TE_TRAIL, PFX_SPRITE, smoke_tex + (rand()%4), (color&~7) + (rand()&7), pt_slowgrav, cl.time,
			cl.time + 3*r_particleduration.value, rand()%7, (rand()%31 - 15), 1, 1, org, 0, 15, 20, 0);
		R_CreateParticle(2, TE_SPIKE, PFX_DART, 0, (color&~7) + (rand()&7), pt_slowgrav, cl.time, cl.time + 0.25, 0, 0, 1, 1, org, 0, 255, 127, 0);
	}
	else if (type == TE_SMALLPUFF)
	{
		R_CreateParticle(2, TE_TRAIL, PFX_SPRITE, smoke_tex + (rand()%4), (color&~7) + (rand()&7), pt_slowgrav, cl.time,
			cl.time + 3*r_particleduration.value, rand()%7, (rand()%31 - 15), 1, 1, org, 0, 15, 40, 0);
	}
	else if (type == TE_GLOWPUFF)
	{
		R_CreateParticle(count, TE_GLOWPUFF, PFX_GLOW, 0, 107  + (rand()&3), pt_slowgrav, cl.time,
				cl.time + 0.1*(rand()%5), 0, 0, 0, 0, org, 0, 8, 20, 32);
	}
	else if (type == TE_BLOOD)
	{
		R_CreateParticle(count, TE_BLOOD, PFX_SPRITE, blood_tex + (rand()%4), (color&~7) + (rand()&7), pt_slowgrav, cl.time,
			cl.time + 0.25, rand()%7, (rand()%31 - 15), 1, 1, org, 0, 63, 0, 0);
	}
	else
		R_CreateParticle(count, type, PFX_GLOW, 0, (color&~7) + (rand()&7), pt_slowgrav, cl.time,
			cl.time + 0.1*(rand()%5), 0, 0, 1, 1, org, 0, 50, 0, 0);
}


/* === Shockwaves === */
void R_Shock1 (vec3_t org, int type)
{
	int			i, j, count;
	particle_t	*p;

	if (!free_particles)
		return;
	p = free_particles;
	free_particles = p->next;
	p->next = active_particles;
	active_particles = p;
	p->die = cl.time + 1;
	p->drawtype = PFX_GLOW;
	if (type == 1)
		p->fxtype = TE_SHOCKWAVE;
	else if (type == 2)
	{
		p->fxtype = TE_BIGSHOCKWAVE;
		p->die = cl.time + 0.125;
	}
	else
		p->fxtype = TE_SPARKWAVE;
	p->color = 7 + (rand()&7);
	p->type = pt_slowgrav;
	p->born = cl.time;
	for (j=0 ; j<3 ; j++)
	{
		p->org[j] = org[j];
		p->vel[j] = 0;
	}
}

/* === Rain and Snow === */
void R_RainParticles (vec3_t org, int radius, int type)
{
	int			i, j, num;
	particle_t	*p;

#ifdef GLQUAKE		
	num = 16;
#else
	num = 32;
#endif

	num = num * r_particledensity.value;	
	for (i=0 ; i<num ; i++)
	{
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;
		p->born = cl.time;
		p->drawtype = PFX_GLOW;
		if (type == 1)
		{
			p->fxtype = TE_RAIN;
			p->die = cl.time + 2;
		}
		else
		{
			p->fxtype = TE_SNOW;
			p->die = cl.time + 4;
		}

		p->type = pt_blob;
		p->color = 66 + rand()%6;
		for (j=0 ; j<2 ; j++)
		{
			p->org[j] = org[j] + (rand()%radius) - (radius>>1);
			p->vel[j] = 0;
		}
		p->org[2] = org[2] + (rand()%32);

		if (type == 1)
			p->vel[2] = -400;
		else
			p->vel[2] = -100;
	}
}

/*==========
R_LavaSplash
==========*/
void R_LavaSplash (vec3_t org)
{
	int			i, j, k, count;
	particle_t	*p;
	float		vel;
	vec3_t		dir;

#ifdef GLQUAKE
	count = 4;
#else
	count = 16;
#endif
	count = count * r_particledensity.value;

	for (i=-count ; i<count ; i++)
		for (j=-count ; j<count ; j++)
			for (k=0 ; k<1 ; k++)
			{
				if (!free_particles)
					return;
				p = free_particles;
				free_particles = p->next;
				p->next = active_particles;
				active_particles = p;
		
				p->die = cl.time + 1;
				p->born = cl.time;
				p->color = 224 + (rand()&7);
				p->type = pt_slowgrav;
				p->fxtype = TE_LAVASPLASH;
				p->drawtype = PFX_TWINKLE;
				
				dir[0] = j*32 + (rand()&7);
				dir[1] = i*32 + (rand()&7);
				dir[2] = (rand()&20);
	
				p->org[0] = org[0] + (dir[0] * worldscale.value);
				p->org[1] = org[1] + (dir[1] * worldscale.value);
				p->org[2] = org[2] + ((rand()&63) * worldscale.value);
	
				VectorNormalize (dir);						
				vel = 150 + (rand()&63);
				VectorScale (dir, vel, p->vel);
			}

	for (i=-count ; i<count ; i++)
		for (j=-count ; j<count ; j++)
			for (k=0 ; k<1 ; k++)
			{
				if (!free_particles)
					return;
				p = free_particles;
				free_particles = p->next;
				p->next = active_particles;
				active_particles = p;
		
				p->die = cl.time + 1;
				p->born = cl.time;
				p->color = 224 + (rand()&7);
				p->type = pt_slowgrav;
				p->fxtype = TE_LAVASPLASH;
				p->drawtype = PFX_TWINKLE;
				
				dir[0] = j*8 + (rand()&7);
				dir[1] = i*8 + (rand()&7);
				dir[2] = 512;
	
				p->org[0] = org[0] + (dir[0] * worldscale.value);
				p->org[1] = org[1] + (dir[1] * worldscale.value);
				p->org[2] = org[2] + ((rand()&63) * worldscale.value);
	
				VectorNormalize (dir);						
				vel = 150 + (rand()&63);
				VectorScale (dir, vel, p->vel);
			}
}

/*==============
R_TeleportSplash
==============*/
void R_TeleportSplash (vec3_t org)
{
	R_CreateParticle(5, TE_TELEPORT, PFX_SPRITE, spark_tex, 7 + (rand()&7), pt_slowgrav, cl.time,
		cl.time + 0.5, rand()%7, (rand()%6)-3, 1, 1, org, 7, 0, 63, 63);
}

/*====
R_Fire
====*/
void R_Fire (vec3_t org, float power)
{
	int			i;
	vec3_t		vel;
	particle_t	*p;
	
	for (i=0 ; i<power ; i++)
	{
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 0.5;
		p->born = cl.time;
		p->color = 224 + (rand()&7);
		p->type = pt_slowgrav;
		p->fxtype = TE_FIRE;
		p->drawtype = PFX_SPRITE;
		p->tex_num = fire_tex + (rand()%6);
		p->angle = 0;
		p->v_angle = 0;
		
		p->org[0] = org[0] + ((((rand()%32) - 16) * power) * worldscale.value);
		p->org[1] = org[1] + ((((rand()%32) - 16) * power) * worldscale.value);
		p->org[2] = org[2] + (((rand()&16) - 16)  * worldscale.value);

		p->vel[0] = ((rand()%32) - 16);
		p->vel[1] = ((rand()%32) - 16);
		p->vel[2] = (rand()%16 + 8) * (power * 32);
	}
}

/*=====
R_Radar
=====*/
void R_Radar (vec3_t org, vec3_t dir)
{
	R_CreateParticle(1, TE_RADARBLIP, PFX_HUD, radar_tex, 224 + (rand()&7), pt_static, cl.time,
			cl.time + 10, 0, 1.57, 1, 1, org, 0, 0, 0, 0);
}

/*===========
R_RocketTrail
===========*/
void R_RocketTrail (vec3_t start, vec3_t end, int type)
{
	vec3_t		vec;
	float		len;
	int			j;
	particle_t	*p;
	int			dec;
	static int	tracercount;

	VectorSubtract (end, start, vec);
	len = VectorNormalize (vec);
	if (type < 128)
		dec = 10/r_particledensity.value;
	else
	{
		dec = 1;
		type -= 128;
	}
	while (len > 0)
	{
		len -= dec;

		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;
		
		VectorCopy (vec3_origin, p->vel);

		switch (type)
		{
			case 0:	// rocket trail
				R_CreateParticle(1, TE_THRUST, PFX_SPRITE, explo_tex + (rand()%4), ramp3[(int)p->ramp], pt_fire, cl.time,
							 	 cl.time + 3 * r_particleduration.value, rand()%7, (rand()%30)-15, 1, 1, start, 2, 25, 0, 0);
				break;

			case 1:	// smoke 
				R_CreateParticle(1, TE_TRAIL, PFX_SPRITE, smoke_tex + (rand()%4), (rand()&3) + 2, pt_fire, cl.time,
							 	 cl.time + 3 * r_particleduration.value, rand()%7, (rand()%6)-3, 1, 1, start, 0, 10, 0, 0);
				break;

			case 2:	// blood
				R_CreateParticle(1, TE_BLOODTRAIL, PFX_SPRITE, smoke_tex + (rand()%4), 67 + (rand()&3), pt_grav, cl.time,
							 	 cl.time + 1, rand()%7, (rand()%6)-3, 1, 1, start, 3, 5, 0, 0);
				len -= (50/r_particledensity.value);
				break;

			case 3:
			case 5:	// tracer
				p->type = pt_static;
				if (type == 3 && fh1)
				{
					p->die = cl.time + 0.5 * r_particleduration.value;
					p->born = cl.time;
					p->fxtype = TE_WIZSPIKE;
					p->drawtype = PFX_GLOW;
					p->color = 230 + ((tracercount&4)<<1);
					for (j=0; j<3 ; j++)
						p->vel[j] = (end[j] - start[j]) * 20;
					VectorCopy (start, p->org);
				}
				else if (type == 3 && !fh1)
				{
					p->die = cl.time + 1 * r_particleduration.value;
					p->born = cl.time;
					p->fxtype = TE_WIZSPIKE;
					p->drawtype = PFX_GLOW;
					p->color = 230 + ((tracercount&4)<<1);
					VectorCopy (start, p->org);
				}
				else if (type != 3 && fh1)
				{
					p->die = cl.time + 0.5 * r_particleduration.value;
					p->born = cl.time;
					p->fxtype = TE_KNIGHTSPIKE;
					p->drawtype = PFX_GLOW;
					p->color = 230 + ((tracercount&4)<<1);
					VectorCopy (start, p->org);
					for (j=0 ; j<3 ; j++)
						p->org[j] = start[j] + (((rand()&20)-10) * worldscale.value);
				}
				else if (type != 3 && !fh1)
				{
					p->die = cl.time + 1 * r_particleduration.value;
					p->born = cl.time;
					p->fxtype = TE_KNIGHTSPIKE;
					p->drawtype = PFX_GLOW;
					p->color = 230 + ((tracercount&4)<<1);
					VectorCopy (start, p->org);
				}
				tracercount++;
				break;

			case 4:	// slight blood
				p->fxtype = TE_BLOODTRAIL;
				p->drawtype = PFX_SPRITE;
				p->tex_num = smoke_tex;
				p->die = cl.time + 1;
				p->born = cl.time;
				p->type = pt_grav;
				p->color = 67 + (rand()&3);
				for (j=0 ; j<3 ; j++)
				{
					p->org[j] = start[j] + (((rand()%6)-3) * worldscale.value);
					p->vel[j] = (rand()%10)-5;
				}
				len -= (50/r_particledensity.value);
				break;

			case 6:	// voor trail
				p->fxtype = TE_SPARK;
				p->drawtype = PFX_GLOW;
				p->born = cl.time;
				p->color = 9*16 + 8 + (rand()&3);
				p->type = pt_static;
				p->die = cl.time + 0.25;
				for (j=0 ; j<3 ; j++)
					p->org[j] = start[j] + (((rand()&3)-6) * worldscale.value);
				break;
		}
		VectorAdd (start, vec, start);
	}
}

extern	cvar_t	sv_gravity;

/*=================
R_DrawDartParticles
=================*/
void R_DrawDartParticle(particle_t *p, float age, float ttlive, float distance, float grav, float frametime)
{
	int			i, j, bob;
	float		radius;
	float		cutoff;
	float		intensity;
	float		rander;
	float		red, green, blue;
	vec3_t		v, offset;
	vec3_t		forward, right, up;
	
	bob = p->fxtype;
	switch(bob)
	{
		case TE_SUPERSPIKE:
		case TE_SPIKE:
			radius = 1;
			intensity = ttlive * 4;
			red = intensity * 0.8f;
			green = intensity * 0.4f;
			blue = 0.1;
			cutoff = 1;
			p->vel[2] -= grav * 16;
			break;
	}

	glColor3f(red * over_dim, green * over_dim, blue * over_dim);
	
	if (radius <= 0.0f) radius = 1.0f;

	radius *= worldscale.value;
	cutoff *= worldscale.value;

	VectorScale(p->vel, -0.0625, offset);
	AngleVectors (offset, forward, right, up);
	VectorAdd(offset, p->org, offset);

	if (distance > cutoff)
	{
		glBegin(GL_TRIANGLE_FAN);
		for (i=0 ; i<3 ; i++)
			v[i] = p->org[i] - vpn[i] * radius;
		glVertex3fv(v);
		glColor3f(0.0f, 0.0f, 0.0f);
		for (i=16; i>=0; i--)
		{
			float a = i/16.0f * M_PI * 2;
			for (j=0; j<3; j++)
				v[j] = offset[j] + right[j]*cos(a)*radius + up[j]*sin(a)*radius;
			glVertex3fv(v);
		}
		glEnd();
	}
}

/*=================
R_DrawGlowParticles
=================*/
void R_DrawGlowParticle(particle_t *p, float age, float ttlive, float distance, float grav, float frametime)
{
	int			i, j, bob;
	float		radius;
	float		cutoff;
	float		intensity;
	float		rander;
	float		red, green, blue;
	vec3_t		v;
	
	bob = p->fxtype;
	switch(bob)
	{
		case TE_WIZSPIKE:
			radius = 5 + ttlive * 10;
			intensity = ttlive;
			red = 0.4f * intensity;
			green = 0.7f* intensity;
			blue = 0.1;
			cutoff = radius;
			break;

		case TE_KNIGHTSPIKE:
			radius = 5 + ttlive * 10;
			intensity = ttlive;
			red = intensity * 0.8f;
			green = intensity * 0.4f;
			blue = 0.1;
			cutoff = radius;
			break;

		case TE_SPARK:
			radius = 16;
			intensity = ttlive * 4;
			red = intensity * 0.8f;
			green = intensity * 0.4f;
			blue = 0.1;
			cutoff = radius;
			break;

		case TE_TRAIL:
			radius = 10 + (2*r_particleduration.value - ttlive) * 5;
			intensity = ttlive * 0.1;
			red = green = blue = 0.2f * intensity;
			cutoff = radius * 5;
			break;

		case TE_FREE:
			radius = 1;
			glColor3ubv ((byte *)&d_8to24table[(int)p->color]);
			p->vel[2] -= grav * 32;
			cutoff = radius * 5;
			break;

		case TE_GUNSHOT:
			radius = 8;
			intensity = ttlive * 2;
			red = green = blue = intensity * 0.25f;
			cutoff = radius * 5;
			break;

		case TE_GLOWPUFF:
			radius = 12;
			intensity = ttlive * 2;
			red = intensity * 0.8f;
			green = intensity * 0.4f;
			blue = intensity * 0.1f;
			cutoff = radius * 5;
			break;

		case TE_SMALLPUFF:
			radius = 8 + (4 * r_particleduration.value - ttlive) * 4;
			intensity = ttlive * 0.1;
			red = green = blue = intensity * 0.2f;
			p->vel[2] -= grav * 0.5;
			cutoff = radius * 5;
			break;
		
		case TE_SUPERSPIKE:
			radius = 8;
			intensity = ttlive * 2;
			red = green = blue = intensity * 0.25f;
			cutoff = radius * 5;
			break;

		case TE_EXPLOSION:
			radius = 100 + (255 - ttlive * 255);
			//radius = 30 + ttlive * 2048;
			intensity = ttlive;
			red = intensity * 0.8f;
			green = intensity * 0.4f;
			blue = intensity * 0.1f;
			cutoff = radius;
			break;

		case TE_TAREXPLOSION:
			radius = 30 - (ttlive * 6);
			intensity = ttlive * 0.2;
			red = intensity * 0.8f;
			green = intensity * 0.1f;
			blue = intensity * 0.6f;
			cutoff = radius * 5;
			break;

		case TE_LAVASPLASH:
			radius = 30 + ((1 - ttlive) * 40);
			intensity = 1 - age;
			red = intensity * 0.8f;
			green = intensity * 0.4f;
			blue = 0.1f;
			p->vel[2] -= grav * 8;
			cutoff = radius * 5;
			break;

		case TE_TELEPORT:
			radius = 10 - (ttlive * 0.1);
			intensity = ttlive * 3;
			red = green = blue = intensity * 1.0f;
			cutoff = radius * 5;
			break;

		case TE_EXPLOSION2:
			radius = 30 - (ttlive * 6);
			intensity = 1 - (age * 0.2); //ttlive * 0.2;
			red = intensity * 0.8f;
			green = intensity * 0.4f;
			blue = 0.1f;
			cutoff = radius * 5;
			break;

		case TE_BRIGHTFIELD:
			radius =  10;
			intensity = ttlive * 5;
			red = green = blue = intensity * 0.9f;
			cutoff = radius * 5;
			break;

		case TE_SMALLGLOW:
			radius =  20;
			red = 0.7f;
			green = 0.55f;
			blue = 0.4f;
			cutoff = radius * 5;
			break;

		case TE_BIGGLOW:
			radius =  50;
			red = 0.7f;
			green = 0.55f;
			blue = 0.4f;
			cutoff = radius * 5;
			break;

		case TE_BUBBLEGLOW:
			radius = 5;
			red = green = blue = 0.8f;
			cutoff = radius;
			break;
	
		case TE_SPARKGLOW:
			radius =  20;
			red = 0.7f;
			green = 0.55f;
			blue = 0.4f;
			cutoff = radius * 5;
			break;

		case TE_BOLTGLOW:
			radius =  20;
			red = 0.02f;
			green = 0.0f;
			blue = 0.1f;
			cutoff = radius;
			break;

		case TE_SHOCKWAVE:
			radius = 50 + (1 - ttlive) * 255;
			intensity = ttlive;
			red = intensity * 0.7f;
			green = intensity * 0.55f;
			blue = 0.4f;
			cutoff = radius;
			break;

		case TE_BIGSHOCKWAVE:
			radius = 100 + (255 - ttlive * 2047);
			glColor3f(0.35f, 0.24f, 0.2f);
			red = 0.35f;
			green = 0.24f;
			blue = 0.2f;
			cutoff = radius;
			break;

		case TE_SPARKWAVE:
			radius = 20 + (1 - ttlive) * 127;
			intensity = ttlive;
			red = intensity * 0.8f;
			green = intensity * 0.4f;
			blue = 0.1f;
			cutoff = radius;
			break;

		case TE_RAIN:
			radius =  2;
			intensity = (2-ttlive)*0.5;
			red = intensity * 0.2f;
			green = intensity * 0.2f;
			blue = 0.4f;
			p->vel[2] -= grav * 8;
			p->vel[1] = p->vel[1] + ((rand()%4)-2);
			p->vel[0] = p->vel[0] + ((rand()%4)-2);
			cutoff = radius * 5;
			break;
	
		case TE_SNOW:
			radius =  3;
			intensity = (4-ttlive)*0.5;
			red = intensity * 0.9f;
			green = intensity * 0.9f;
			blue = 0.9f;
			p->vel[2] -= grav * 2;
			p->vel[1] = p->vel[1] + ((rand()%16)-8);
			p->vel[0] = p->vel[0] + ((rand()%16)-8);
			cutoff = radius * 5;
			break;

		case TE_GOLDGLOW:
			radius =  35;
			red = 0.9f;
			green = 0.4f;
			blue = 0.1f;
			cutoff = radius;
			break;

		case TE_BLUEGLOW:
			radius =  35;
			red = 0.0f;
			green = 0.0f;
			blue = 0.5f;
			cutoff = radius;
			break;

		case TE_TORCHGLOW:
			radius = 30;
			intensity = 0.5 - ( (2000 - distance) * 0.00025 );
			red = intensity * 0.8f;
			green = intensity * 0.4f;
			blue = 0.1f;
			cutoff = radius;
			break;
	}

	if (bob != TE_FREE)
		glColor3f(red * over_dim, green * over_dim, blue * over_dim);
	
	if (radius <= 0.0f) radius = 1.0f;

	radius *= worldscale.value;
	cutoff *= worldscale.value;

	if (distance > cutoff)
	{
		glBegin(GL_TRIANGLE_FAN);
		for (i=0 ; i<3 ; i++)
			v[i] = p->org[i] - vpn[i] * radius;
		glVertex3fv(v);
		glColor3f(0.0f, 0.0f, 0.0f);
		for (i=16; i>=0; i--)
		{
			float a = i/16.0f * M_PI * 2;
			if (p->drawtype == PFX_TWINKLE)
			{
				rander = radius + radius * ((rand ()&0x7fff) / ((float)0x7fff));
				for (j=0; j<3; j++)
				 //v[j] =	p->org[j] + vright[j]*cos(a)*radius + vup[j]*sin(a)*radius;
				 v[j] =	p->org[j] + vright[j]*cos(a)* rander + vup[j]*sin(a)*rander;
			}
			else
			{
				for (j=0; j<3; j++)
					v[j] = p->org[j] + vright[j]*cos(a)*radius + vup[j]*sin(a)*radius;
			}
			glVertex3fv(v);
		}
		glEnd();
	}
}

/*==================
R_DrawSpriteParticle
==================*/
void R_DrawSpriteParticle(particle_t *p, float age, float ttlive, float distance, float grav, float frametime)
{
	float		size, red, green, blue, scale, limit, angle;
	vec3_t		up, right, point;
	int			i,j, bob;

	bob = p->fxtype;
	switch(bob)
	{
		default:
		case TE_TRAIL:
			p->scale = 20 + (3*r_particleduration.value - ttlive) * 7;
			p->alpha = ttlive * 0.1;
			p->v_angle = p->v_angle * 0.93;
			red = green = blue = over_dim;
			break;

		case TE_THRUST:
			p->scale = 5 + (6*r_particleduration.value - ttlive) * 3;
			p->alpha = ttlive * 0.1;
			red = green = blue = (1 - (cl.time - p->born));
			p->v_angle *= 0.95f;
			VectorScale(p->vel, 1 - (0.85 * frametime), p->vel);
			p->vel[2] += 2 * frametime;
			break;

		case TE_EXPLOSION:
			p->scale = 20 + ((5 - ttlive) * 40);
			p->alpha = ttlive * 0.25;
			red = green = blue = (1 - (cl.time - p->born));
			p->v_angle *= 0.95f;
			VectorScale(p->vel, 0.97f, p->vel);
			break;

		case TE_FIRE:
			p->scale = 5 + (3*r_particleduration.value - ttlive) * 5;
			p->alpha = 1 - age * 2;
			p->vel[2] = p->vel[2] + 15;
			red = green = blue = over_dim;
			break;

		case TE_BLOODTRAIL:
			p->scale = 5 + (1 - ttlive) * 15;
			p->alpha = ttlive;
			red = over_dim * 0.5;
			green = blue = 0;
			break;
		
		case TE_RADARBLIP:
			p->scale = 32;
			p->alpha = ttlive * 0.1f;
			red = green = blue = over_dim;
			break;

		case TE_GUNSHOT:
			p->alpha = ttlive;
			red = green = blue = over_dim;
			p->vel[2] -= grav * 8;
			break;

		case TE_BLOOD:
			p->scale = 2 + (0.25 - ttlive) * 32;
			p->alpha = ttlive * 4;
			p->v_angle = p->v_angle * 0.93;
			red = green = blue = over_dim;
			break;

		case TE_TELEPORT:
			p->scale = 20 + (0.5 - ttlive) * 60;
			p->alpha = ttlive * 2;
			red = green = blue = over_dim;
			break;
	}

	VectorCopy(vright, right);
	VectorCopy(vup, up);

	// fudge size of nearby particles and scale by world
	size = p->scale * worldscale.value;
	limit = size * 5 + 1;
	if (distance < limit && bob != TE_FIRE)
		size = size * (distance/limit);

	// draw particle
	glColor4f(red, green, blue, p->alpha);
	GL_Bind(p->tex_num);
	glBegin (GL_QUADS);

	for (i=4; i>=0; i--)
	{
		if (i == 1)
			glTexCoord2f (0, 0);
		else if (i == 2)
			glTexCoord2f (0, 1);
		else if (i == 3)
			glTexCoord2f (1, 1);
		else if (i == 4)
			glTexCoord2f (1, 0);
		else
			glTexCoord2f (1, 0);

		angle = rotater[i] + p->angle;
		for (j=0; j<3; j++)
			point[j] = p->org[j] + vright[j]*cos(angle)*size + vup[j]*sin(angle)*size;
		glVertex3fv(point);
	}
	glEnd ();
}

#ifdef GLQUAKE
void R_DrawParticles (void)
{
	particle_t		*p, *kill;
	float			grav;
	int				i, j, bob, lastcase;
	float			time2, time3;
	float			time1;
	float			dvel;
	float			frametime;
	float			ttlive;
	float			alpha;
	float			age;
	float			rander;
	
	vec3_t			up, right;
	float			scale;

	vec3_t	v;
	float		radius;
	float		distance, cutoff;
	float		intensity;
	
	if (gl_fogenable.value)
		glDisable(GL_FOG);
	glDepthMask (FALSE);
	glDisable (GL_TEXTURE_2D);
	glShadeModel (GL_SMOOTH);
	glEnable (GL_BLEND);
	glBlendFunc (GL_ONE, GL_ONE);
	glPushMatrix();			

	VectorScale (vup, 1, up);
	VectorScale (vright, 1, right);

	frametime = cl.time - cl.oldtime;
	time3 = frametime * 15;
	time2 = frametime * 10; // 15;
	time1 = frametime * 5;
	grav = frametime * sv_gravity.value * 0.05;
	dvel = 4*frametime;
	
	for ( ;; ) 
	{
		kill = active_particles;
		if (kill && kill->die < cl.time)
		{
			active_particles = kill->next;
			kill->next = free_particles;
			free_particles = kill;
			continue;
		}
		break;
	}

	lastcase = PFX_GLOW;

	for (p=active_particles ; p ; p=p->next)
	{
		for ( ;; )
		{
			kill = p->next;
			if (kill && kill->die < cl.time)
			{
				p->next = kill->next;
				kill->next = free_particles;
				free_particles = kill;
				continue;
			}
			break;
		}

		// time to live per particle
		ttlive = p->die - cl.time;
		if (ttlive > 5 * r_particleduration.value)
			p->die = cl.time + 0.01f;
		age = cl.time - p->born;
		VectorSubtract(p->org, r_origin, v);
		distance = Length(v);

		bob = p->drawtype;
		switch(bob)
		{
			case PFX_HUD:
				if (lastcase != PFX_HUD)
				{
					glDisable (GL_DEPTH_TEST);
					glEnable (GL_TEXTURE_2D);
					glEnable (GL_BLEND);
					//glBlendFunc (GL_SRC_ALPHA, GL_ONE);
					glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
					glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
					lastcase = PFX_SPRITE;
				}
				R_DrawSpriteParticle(p, age, ttlive, distance, grav, frametime);
				break;

			case PFX_SPRITE:
				if (lastcase != PFX_SPRITE)
				{
					//if (lastcase == PFX_HUD)
					//glEnable (GL_DEPTH_TEST);
					glEnable (GL_TEXTURE_2D);
					glEnable (GL_BLEND);
					glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
					glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
					lastcase = PFX_SPRITE;
				}
				glEnable (GL_DEPTH_TEST);
				R_DrawSpriteParticle(p, age, ttlive, distance, grav, frametime);
				break;

			case PFX_DART:
				if (lastcase != PFX_DART)
				{
					glEnable (GL_DEPTH_TEST);
					glDisable (GL_TEXTURE_2D);
					glEnable (GL_BLEND);
					glBlendFunc (GL_ONE, GL_ONE);
					glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
					lastcase = p->drawtype;
				}
				R_DrawDartParticle(p, age, ttlive, distance, grav, frametime);
				break;
			
			case PFX_GLOW:
			case PFX_TWINKLE:
			default:
				if (lastcase != PFX_GLOW || lastcase != PFX_TWINKLE)
				{
					//if (lastcase == PFX_HUD)
						glEnable (GL_DEPTH_TEST);
					glDisable (GL_TEXTURE_2D);
					glEnable (GL_BLEND);
					glBlendFunc (GL_ONE, GL_ONE);
					glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
					lastcase = p->drawtype;
				}
				R_DrawGlowParticle(p, age, ttlive, distance, grav, frametime);
				break;
		}
		p->org[0] += p->vel[0]*frametime*worldscale.value;
		p->org[1] += p->vel[1]*frametime*worldscale.value;
		p->org[2] += p->vel[2]*frametime*worldscale.value;

		p->angle += (p->v_angle * frametime);
		p->angle = fmod(p->angle, 360.0f);
	}

	glPopMatrix();
	glColor3f (1,1,1);
	glDisable (GL_BLEND);
	glEnable (GL_TEXTURE_2D);
	glEnable (GL_DEPTH_TEST);
	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glDepthMask (TRUE);
	if (gl_fogenable.value)
		glEnable(GL_FOG);
}
#endif

#ifndef GLQUAKE
/*===============
R_DrawParticles
===============*/
void R_DrawParticles (void)
{
	particle_t		*p, *kill;
	float			grav;
	int				i;
	float			time2, time3;
	float			time1;
	float			dvel;
	float			frametime;
	
	D_StartParticles ();

	VectorScale (vright, xscaleshrink, r_pright);
	VectorScale (vup, yscaleshrink, r_pup);
	VectorCopy (vpn, r_ppn);
	frametime = cl.time - cl.oldtime;
	time3 = frametime * 15;
	time2 = frametime * 10; // 15;
	time1 = frametime * 5;
	grav = frametime * sv_gravity.value * 0.05;
	dvel = 4*frametime;
	
	for ( ;; ) 
	{
		kill = active_particles;
		if (kill && kill->die < cl.time)
		{
			active_particles = kill->next;
			kill->next = free_particles;
			free_particles = kill;
			continue;
		}
		break;
	}

	for (p=active_particles ; p ; p=p->next)
	{
		for ( ;; )
		{
			kill = p->next;
			if (kill && kill->die < cl.time)
			{
				p->next = kill->next;
				kill->next = free_particles;
				free_particles = kill;
				continue;
			}
			break;
		}

		D_DrawParticle (p);
		p->org[0] += p->vel[0]*frametime;
		p->org[1] += p->vel[1]*frametime;
		p->org[2] += p->vel[2]*frametime;

		p->angle += p->v_angle;
		p->angle = fmod(p->angle, 1.0f);

		switch (p->type)
		{
		case pt_static:
			break;
		case pt_fire:
			p->ramp += time1;
			if (p->ramp >= 6)
				p->ramp = 6;
			else
				p->color = ramp3[(int)p->ramp];
			//p->vel[2] += grav;
			break;

		case pt_explode:
			p->ramp += time2;
			if (p->ramp >=8)
				p->die = -1;
			else
				p->color = ramp1[(int)p->ramp];
			for (i=0 ; i<3 ; i++)
				p->vel[i] += p->vel[i]*dvel;
			p->vel[2] -= grav;
			break;

		case pt_explode2:
			p->ramp += time3;
			if (p->ramp >=8)
				p->die = -1;
			else
				p->color = ramp2[(int)p->ramp];
			for (i=0 ; i<3 ; i++)
				p->vel[i] -= p->vel[i]*frametime;
			p->vel[2] -= grav;
			break;

		case pt_blob:
			for (i=0 ; i<3 ; i++)
				p->vel[i] += p->vel[i]*dvel;
			p->vel[2] -= grav;
			break;

		case pt_blob2:
			for (i=0 ; i<2 ; i++)
				p->vel[i] -= p->vel[i]*dvel;
			p->vel[2] -= grav;
			break;

		case pt_slowgrav:
			p->vel[2] -= (grav * 0.25);
			break;
		}
	}
	D_EndParticles ();
}
#endif
