/**
 * qml_parser.c
 **/

#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <stdio.h>

#include "qml.h"

#pragma	warning (disable:4244)
#pragma warning (disable:4996)

qmlParse_t	qmlTokens[]	=	{
	// operators
	{TK_EQUALS,			ttOperator,	qOpEq,					0},
	{TK_NOTEQUALS,		ttOperator,	qOpNotEq,				0},
	{TK_GREATEROREQUAL,	ttOperator,	qOpGtEq,				0},
	{TK_LESSEROREQUAL,	ttOperator,	qOpLtEq,				0},
	// control
	// FIXME: use enums instead the magic numbers
	{TK_QMLBEGIN,		ttCtrl,		qCtlQMLBegin,			0},
	{TK_QMLEND,			ttCtrl,		qCtlQMLEnd,				0},
	{TK_BEGINTAG,		ttCtrl,		qCtlTagBegin,			0},
	{TK_ENDTAG,			ttCtrl,		qCtlTagEnd,				0},
	{TK_BEGINVAR,		ttCtrl,		qCtlVarBegin,			0},
	{TK_ENDVAR,			ttCtrl,		qCtlVarEnd,				0},
	{TK_BEGINVALUE,		ttCtrl,		qCtlValBegin,			0},
	// these operators were moved here for not conflicting with control tokens
	{TK_GREATER,		ttOperator,	qOpGt,					0},
	{TK_LESSER,			ttOperator,	qOpLt,					0},
	// tags
	// TODO: should define a bitmask of required attributes
	{TK_TXT,			ttTag,		qtText,					0},
	{TK_PIC,			ttTag,		qtPic,					0},
	{TK_BSP,			ttTag,		qtBsp,					0},
	{TK_MDL,			ttTag,		qtMdl,					0},
	{TK_SPR,			ttTag,		qtSpr,					0},
	{TK_IF,				ttTag,		qtIf,					0},
	{TK_ENDIF,			ttTag,		qtEndif,				0},
	{TK_BAR,			ttTag,		qtBar,					0},
	{TK_SND,			ttTag,		qtSnd,					0},
	// attributes
	{TK_X,				ttAttr,		qaX,					DT_INT|DT_VAR|DT_PERCENTUAL},
	{TK_YAW,			ttAttr,		qaYaw,					DT_ANGLE|DT_VAR|DT_PERCENTUAL},
	{TK_Y,				ttAttr,		qaY,					DT_INT|DT_VAR|DT_PERCENTUAL},
	{TK_SRC,			ttAttr,		qaSrc,					DT_STRING},
	{TK_VALUE,			ttAttr,		qaValue,				DT_STRING},
	{TK_SCALE,			ttAttr,		qaScale,				DT_FLOAT|DT_PERCENTUAL},
	{TK_FRAME,			ttAttr,		qaFrame,				DT_BYTE|DT_INT},
	{TK_SKIN,			ttAttr,		qaSkin,					DT_BYTE|DT_INT},
	{TK_COLORMAP,		ttAttr,		qaColorMap,				DT_BYTE|DT_INT},
	{TK_ROLL,			ttAttr,		qaRoll,					DT_ANGLE|DT_VAR|DT_PERCENTUAL},
	{TK_TILT,			ttAttr,		qaTilt,					DT_ANGLE|DT_VAR|DT_PERCENTUAL},
	{TK_WIDTH,			ttAttr,		qaWidth,				DT_INT|DT_VAR|DT_PERCENTUAL},
	{TK_HEIGHT,			ttAttr,		qaHeight,				DT_INT|DT_VAR|DT_PERCENTUAL},
	{TK_RED,			ttAttr,		qaRed,					DT_BYTE|DT_FLOAT|DT_VAR|DT_PERCENTUAL},
	{TK_GREEN,			ttAttr,		qaGreen,				DT_BYTE|DT_FLOAT|DT_VAR|DT_PERCENTUAL},
	{TK_BLUE,			ttAttr,		qaBlue,					DT_BYTE|DT_FLOAT|DT_VAR|DT_PERCENTUAL},
	{TK_ALPHA,			ttAttr,		qaAlpha,				DT_BYTE|DT_FLOAT|DT_VAR|DT_PERCENTUAL},
	{TK_VOLUME,			ttAttr,		qaVolume,				DT_FLOAT|DT_VAR|DT_PERCENTUAL},
	{TK_LOOP,			ttAttr,		qaLoop,					DT_BOOL|DT_INT},
	// environment vars
	{TK_HEALTH,			ttVar,		qvHealth,				DT_INT|DT_PERCENTUAL|DT_STRING},
	{TK_FRAGS,			ttVar,		qvFrags,				DT_INT|DT_STRING},
	{TK_CURRWEP,		ttVar,		qvCurrentWeapon,		DT_BYTE|DT_INT|DT_STRING},
	{TK_CURRAMMO,		ttVar,		qvCurrentAmmo,			DT_BYTE|DT_INT|DT_STRING},
	{TK_ARMORVALUE,		ttVar,		qvArmorValue,			DT_INT|DT_STRING},
	{TK_ARMORTYPE,		ttVar,		qvArmorType,			DT_INT|DT_STRING},
	{TK_AMMOARRAY,		ttVarArray,	qvAmmoArray,			DT_INT|DT_STRING},
	{TK_KILLEDMONSTERS,	ttVar,		qvKilledMonsters,		DT_INT|DT_STRING},
	{TK_TOTALMONSTERS,	ttVar,		qvTotalMonsters,		DT_INT|DT_STRING},
	{TK_SECRETSFOUND,	ttVar,		qvSecretsFound,			DT_INT|DT_STRING},
	{TK_TOTALSECRETS,	ttVar,		qvTotalSecrets,			DT_INT|DT_STRING},
	{TK_TIMEDD,			ttVar,		qvTimeDD,				DT_BYTE|DT_INT|DT_STRING},
	{TK_TIMEHH,			ttVar,		qvTimeHH,				DT_BYTE|DT_INT|DT_STRING},
	{TK_TIMEMM,			ttVar,		qvTimeMM,				DT_BYTE|DT_INT|DT_STRING},
	{TK_TIMESS,			ttVar,		qvTimeSS,				DT_BYTE|DT_INT|DT_STRING},
	{TK_ABSTIME,		ttVar,		qvAbsoluteTime,			DT_INT|DT_STRING},
	{TK_MAPNAME,		ttVar,		qvMapName,				DT_STRING},
	{TK_BSPNAME,		ttVar,		qvBspName,				DT_STRING},
	{TK_GAMEDIR,		ttVar,		qvGameDir,				DT_STRING},
	{TK_MODNAME,		ttVar,		qvModName,				DT_STRING},
	{TK_ENGINE,			ttVar,		qvEngine,				DT_STRING},
	{TK_COOP,			ttVar,		qvCoop,					DT_BYTE|DT_INT|DT_STRING},
	{TK_DEATHMATCH,		ttVar,		qvDeathmatch,			DT_BYTE|DT_INT|DT_STRING},
	{TK_COOP,			ttVar,		qvCoop,					DT_BYTE|DT_INT|DT_STRING},
	{TK_MAXPLAYERS,		ttVar,		qvMaxPlayers,			DT_BYTE|DT_INT|DT_STRING},
	{TK_NUMPLAYERS,		ttVar,		qvNumPlayers,			DT_BYTE|DT_INT|DT_STRING},
	{TK_WORLDTYPE,		ttVar,		qvWorldType,			DT_BYTE|DT_INT|DT_STRING},
	{TK_WEPARRAY,		ttVarArray,	qvWeaponArray,			DT_INT|DT_STRING},
	{TK_KEYARRAY,		ttVarArray,	qvKeyArray,				DT_INT|DT_STRING},
	{TK_RUNEARRAY,		ttVarArray,	qvRuneArray,			DT_INT|DT_STRING},
	{TK_POWERUPARRAY,	ttVarArray,	qvPowerupArray,			DT_INT|DT_STRING},
	{TK_NAMEARRAY,		ttVarArray,	qvClientNameArray,		DT_STRING},
	{TK_SCOREARRAY,		ttVarArray,	qvClientScoreArray,		DT_BYTE|DT_INT|DT_STRING},
	{TK_PINGARRAY,		ttVarArray,	qvClientPingArray,		DT_BYTE|DT_INT|DT_STRING},
	{TK_FUNCARRAY,		ttVarArray,	qvFunctionArray,		DT_BYTE|DT_INT|DT_FLOAT|DT_STRING},
	{TK_EVENTARRAY,		ttVarArray,	qvEventArray,			DT_BYTE|DT_INT|DT_FLOAT|DT_STRING},
	// end of array marker
	{NULL,				ttNone,		qtNone,					0}
};


char	*_qmlptr = NULL;
int		qmlParseNext (void)
{
	int		i = 0;
	int		ctrl = 0;

	if (*_qmlptr)
	{
		switch (ctrl)
		{
			case	0:
				while ((*_qmlptr == ' ') ||
					(*_qmlptr == '\t') ||
					(*_qmlptr == '\n') ||
					(*_qmlptr == '\r'))
				{
					_qmlptr++;
				}
				if (*_qmlptr == '\'')
				{
					ctrl = 2;
				}
				else if (*_qmlptr == '"')
				{
					ctrl = 1;
				}
				else
				{
					while (qmlTokens[i].token != NULL)
					{
						if (!_strnicmp (_qmlptr, qmlTokens[i].token, strlen (qmlTokens[i].token)))
						{
							_qmlptr += strlen (qmlTokens[i].token);
							return (i);
						}

						i++;
					}

					return (-1);
				}
			break;
			case	1:
				while ((*_qmlptr) && (*_qmlptr != '"'))
				{
					_qmlptr++;
				}

				if (*_qmlptr)
				{
					ctrl = 0;
				}
				else
				{
					return (-1);
				}
			break;
			case	2:
				while ((*_qmlptr) && (*_qmlptr != '\''))
				{
					_qmlptr++;
				}

				if (*_qmlptr)
				{
					ctrl = 0;
				}
				else
				{
					return (-1);
				}
			break;
		}		
	}

	return (-1);
}

int		qmlParseStart (char *s)
{
	_qmlptr = s;
	return (qmlParseNext ());
}

float	qmlParseFloatValue (void)
{
	char	buf[128];
	char	*src = _qmlptr, *dst = buf;
	int		i;

	if (*_qmlptr)
	{
		memset (buf, '\0', 128);
		while ((*src) &&
			((*src == ' ') ||
			(*src == '\t') ||
			(*src == '\n') ||
			(*src == '\r')))
		{
			src++;
		}

		while ((*src) &&
			(((*src >= '0') && 
			(*src <= '9')) ||
			(*src == '.')))
		{
			*dst = *src;
			src++;
			dst++;
		}
		
		if ((*src) && (*src == '%'))
		{
			i = atoi (buf);
			if (i < 0)
			{
				i = 0;
			}
			else if (i > 100)
			{
				i = 100;
			}

			_qmlptr = src + 1;
			return (float) i/100;
		}

		_qmlptr = src;
		return (atof (buf));
	}

	return (0);
}

int		qmlParseIntValue (void)
{
	return (int)qmlParseFloatValue ();
}

char	*qmlParseStringValue (void)
{
	char	*buf = NULL;
	char	*ptr = _qmlptr;
	char	separator = *_qmlptr;
	int		i;

	do	{
		ptr++;
	} while ((*ptr) && (*ptr != separator));

	if (*_qmlptr)
	{
		i = (int) (ptr - _qmlptr);
		buf = malloc (1 + (sizeof (char)) * i);
		memset (buf, '\0', i + 1);
		memcpy (buf, _qmlptr + 1, i - 1);
		_qmlptr = ptr + 1;
	}

	return (buf);
}

/**
 * tries to guess all valid ways
 * the current value can be parsed
 * (it's up to the caller to decide
 * which format best suits its needs)
 **/
int		qmlCheckValueType (void)
{
	int		result = DT_NONE, cont, i;
	char	buf[256];
	char	*ptr = _qmlptr, *dst = buf;
	float	f;

	while ((*ptr) && (
		(*ptr == ' ') ||
		(*ptr == '\t') ||
		(*ptr == '\n') ||
		(*ptr == '\r')))
	{
		ptr++;
	}

	cont = (*ptr != '\0');
 	memset (buf, '\0', 256);	

	while (cont)
	{
		if (*ptr == '"')
		{
			result = DT_STRING;
			break;
		}
		else if (*ptr == '\'')
		{
			result = DT_STRING;
			break;
		}
		else if (*ptr == '%')
		{
			if (ptr != _qmlptr)
			{
				i = atoi (buf);
				if ((i >= 0) && (i <= 100))
				{
					result = DT_PERCENTUAL|DT_FLOAT;
					break;
				}
				else
				{
					// TODO: report error (invalid percentual)
					result = DT_NONE;
					break;
				}
			}
		}
		else if ((ptr[0] == '$') && (ptr[1] == '{'))
		{
			i = qmlParseNext();
			// restores the pointer
			_qmlptr = ptr;			
			result = DT_VAR | qmlTokens[i].type;
			break;
		}

		if ((*ptr != '.') && ((*ptr < '0') || (*ptr > '9')))
		{
			f = atof (buf);
			result = DT_FLOAT;
			i = atoi (buf);
			if ((f - i) == 0)
			{
				result |= DT_INT;
				if ((i >= 0) && (i < 360))
				{
					result |= DT_ANGLE;
				}

				if ((i >= 0) && (i < 256))
				{
					result |= DT_BYTE;
				}

				if ((i >= 0) && (i < 2))
				{
					result |= DT_BOOL;
				}
			}

			break;
		}

		if (cont)
		{
			*dst = *ptr;
			dst++;
			ptr++;
			cont = (*ptr != '\0');
		}
	}

	return (result);
}


int		qmlParseIndexValue()
{
	int	i;

	while ((*_qmlptr) && (*_qmlptr != '['))
	{
		_qmlptr++;
	}

	i = qmlParseIntValue();
	while ((*_qmlptr) && (*_qmlptr != ']'))
	{
		_qmlptr++;
	}

	return (i);
}

/**
 * 
 **/
int		qmlSetValue (qmlValue_t *value)
{
	int		i, result = DT_NONE;
	char	buf[64];

	if (value)
	{
		// some clean up
		value->type = DT_NONE;
		value->f = 0.0f;
		value->i = 0;
		value->index = 0;
		value->s = NULL;
		value->var = qvNone;
		i = qmlCheckValueType();		
		if (i != DT_NONE)
		{
			result = i;
			value->type = i;
			if (i & DT_VAR)
			{
				qmlParseNext();	// "${"
				i = qmlParseNext();
				qmlParseNext();	// "}"
				value->var = qmlTokens[i].v.var;
				if (qmlTokens[i].type == ttVarArray)
				{
					value->index = qmlParseIndexValue();	
				}
			}
			else if (i & DT_STRING)
			{
				value->s = qmlParseStringValue ();
			}
			else if (i & (DT_BOOL|DT_BYTE|DT_ANGLE|DT_INT))
			{
				value->i = qmlParseIntValue ();
				value->f = (float)value->i;
				sprintf (buf, "%i", value->i);
				value->s = _strdup(buf);
			}
			else if (i & (DT_PERCENTUAL|DT_FLOAT))
			{
				value->f = qmlParseFloatValue ();
				if (i & DT_PERCENTUAL)
				{
					value->i = (int)(100*value->f);
				}
				else
				{
					value->i = (int)value->f;
				}

				sprintf (buf, "%6.2f", value->f);
				value->s = _strdup(buf);
			}
		}
	}

	return (result);
}

/**
 * 
 **/
qmlOp_t	qmlParseOperator (void)
{
	int		i;

	i = qmlParseNext();
	if ((i != -1) && (qmlTokens[i].type == ttOperator))
	{
		return (qmlTokens[i].v.op);
	}

	return (qOpNone);
}
/**
 *
 **/
qml_t	*qmlLoad (char *text)
{
	qml_t		*result = NULL;	
	int			e, i, j, c, *jumpTable = NULL, ifCount = 0, endifCount = 0;
	qmlValue_t	tmpValue;
	qmlOp_t		tmpOp;

	if (text)
	{
		i = qmlParseStart (text);
		c = 0;
		// first pass - element count, jump table and basic validations
		while ((i != -1) && (*_qmlptr))
		{
			if (qmlTokens[i].v.tag == qtIf)
			{
				qmlSetValue (&tmpValue);
				if (tmpValue.type & DT_VAR)
				{
					tmpOp = qmlParseOperator ();
					if (tmpOp != qOpNone)
					{
						qmlSetValue (&tmpValue);
						ifCount++;
						c++;
					}
					else
					{
						// TODO: report error (invalid operator)
						return NULL;
					}
				}
				else
				{
					// TODO: report error (lvalue is not a var)
					return NULL;
				}
			}
			else if (qmlTokens[i].v.tag == qtEndif) 
			{
				endifCount++;
			}
			else if ((qmlTokens[i].type == ttTag) && (qmlTokens[i].v.tag != qtNone))
			{
				c++;
			}
			else if (qmlTokens[i].type == ttAttr)
			{		
				i = qmlParseNext ();
				if ((qmlTokens[i].type == ttCtrl) && (qmlTokens[i].v.ctrl == qCtlValBegin))	
				{										
					qmlSetValue (&tmpValue);
				}
			}

			i = qmlParseNext ();
		}
		
		if (ifCount != endifCount)
		{
			return (NULL);
		}

		result = malloc (sizeof (qml_t));

		if (c == 0)
		{
			// TODO: report error (no QML tags found)
			return (NULL);
		}

		result->numElements = c;
		result->elements = malloc (sizeof (qmlElement_t) * c);
		for (i = 0; i < c; i++)
		{
			result->elements[i].tag = qtNone;
			result->elements[i].value1.type = ttNone;
			result->elements[i].op = qOpNone;
			result->elements[i].value2.type = ttNone;
		}

		if (ifCount > 0)
		{
			jumpTable = malloc (sizeof (int) * ifCount);
		}
		else
		{
			jumpTable = NULL;
		}

		j = 0;
		e = 0;
		i = qmlParseStart (text);
		while ((i != -1) && (*_qmlptr) && (e < result->numElements))
		{
			if (qmlTokens[i].v.tag == qtIf)
			{
				if (jumpTable)
				{
					qmlSetValue (&result->elements[e].value1);
					if (result->elements[e].value1.type & DT_VAR)
					{
						result->elements[e].op = qmlParseOperator ();
						if (result->elements[e].op != qOpNone)
						{
							qmlSetValue (&result->elements[e].value2);
							result->elements[e].jump = -1;
							jumpTable[j++] = e++;
						}
						else
						{
							// TODO: report error (invalid operator)
							return NULL;
						}
					}
					else
					{
						// TODO: report error (lvalue is not a var)
						return NULL;
					}
				}
				else
				{
					// TODO: report error (jump table not initialized)
					return NULL;
				}
			}
			else if ((qmlTokens[i].v.tag == qtEndif) && (j >= 0) && (e < result->numElements))
			{
				if (jumpTable)
				{
					i = jumpTable[--j];
					result->elements[i].jump = e;
				}
				else
				{
					// TODO: report error (jump table not initialized)
					return NULL;
				}
			}
			else if (qmlTokens[i].type == ttCtrl) 
			{
				switch (qmlTokens[i].v.ctrl)
				{
					// TODO: use this to validate malformed tags
					case	qCtlTagBegin:
						result->elements[e].jump = -1;
					break;
					case	qCtlTagEnd:	// end tag
						if (result->elements[e].tag != qtNone)
						{
							e++;
						}
					break;
				}
			}
			else if ((result->elements[e].tag == qtNone) && (qmlTokens[i].type == ttTag) && (qmlTokens[i].v.tag != qtNone))
			{
				result->elements[e].tag = qmlTokens[i].v.tag;
				result->elements[e].pairs = malloc (sizeof (qmlPair_t) * QMLMAX_ATTR);
				for (i = 0; i < QMLMAX_ATTR; i++)
				{
					result->elements[e].pairs[i].attr = qaNone;
					result->elements[e].pairs[i].value.type = ttNone;
				}
			}
			else if (qmlTokens[i].type == ttAttr)
			{		
				c = qmlTokens[i].v.attr - qaNone - 1;
				result->elements[e].pairs[c].attr = qmlTokens[i].v.attr;				
				i = qmlParseNext ();
				if ((qmlTokens[i].type == ttCtrl) && (qmlTokens[i].v.ctrl == qCtlValBegin))	// "="
				{										
					qmlSetValue (&result->elements[e].pairs[c].value);
				}
			}

			i = qmlParseNext ();
		}
	}

	return (result);
}