import struct
import math
from qmdl.anorms import normals, quadrants


# This wrapper function keeps the file reading code tidy
def _struct_read(s, file):
    return s.unpack(file.read(s.size))

# Triangle is entirely or partially on left hand side of skin
# The partial case cannot occur in classic quake mdl files, but a mdl with a
# complex skinmap can have faces which span both sides. Since seam vertices are
# only translated for backfaces, it makes sense to use this as the default.
FACE_FRONT = 1
# Triangle is entirely on right hand side of skin, and any seam vertices
# belonging to this triangle will be translated half a skinwidth right, as
# coordinates for seam vertices record the position for the front faces
FACE_BACK = 0

# note that for winquake compatibility the onseam field must have the value 32
# for a seam vertex (glquake accepts any non-zero value). In the code below
# we test for onseam in the way glquake does, but make sure to always set the
# value in the way winquake will understand

SEAM_FLAG = 32
OFFSEAM_FLAG = 0

class ParseError(BaseException):
    pass


class Mdl:
    """
    Represents the content of a binary .mdl format model for Quake.

    The Mdl class is designed to be a fairly low level representation of
    the .mdl format. The binary format comprises:
        A header
        A list of skins
        A list of UV coordinates
        A list of triangles
        A list of frames
    The Mdl class mirrors this faithfully, and these components are largely
    left independent of each other.

    The triangle class is a good example of the low level philosophy. The model
    format stores triangles as triples of vertex indices, so the Mdl class
    loads these values as integers. We don't create a collection of vertex
    objects and have the triangles references them. This philosophy lets you
    easily script manipulation of the binary format, but means an action like
    deleting a vertex requires careful coding.

    Skins and frames in Quake models can come in two flavours: ungrouped and
    grouped. Ungrouped skins are single images and ungrouped frames are single
    poses of the model. Grouped frames comprise a list of frames along with
    timing data so that the Quake engine can automatically loop the animation;
    Grouped skins work similarly. These flavours are represented by pairs of
    classes.

    From the point of view of the Mdl class, the list of skins/frames is a
    heterogeneous collection which can contain both of the classes. The classes
    are responsible for outputting their own binary representations through the
    "write" method they implement. This means the Mdl class can write all of
    its skins and frames without considering the type of frame or skin it is
    handling.
    
    """

    # STRUCTS
    # A bunch of static objects to avoid repeated construction during loops
    header_struct = struct.Struct("<4sl3f3ff3fllllllllf")
    single_struct = struct.Struct("<l")
    duration_struct = struct.Struct("<f")
    triple_struct = struct.Struct("<3l")
    coord_struct = struct.Struct("<4B")
    name_struct = struct.Struct("<16s")

    def __init__(self):
        """
        Fill in defaults everywhere to avoid AttributeError and create lists

        The values chosen are minimal defaults. Most would not produce a
        meaningful or useful model if not updated
        
        """
        self.ident = b"IDPO"
        self.version = 6
        self.scale = (0, 0, 0)
        self.origin = (0, 0, 0)
        self.boundingradius = 0
        self.eyeposition = (0, 0, 0)
        self.skins = []
        self.skinwidth, self.skinheight = 0, 0
        self.vertices = []
        self.triangles = []
        self.frames = []
        self.synctype = 1
        self.flags = 0
        self.average_size = 0
        self._num_skins = 0
        self._num_verts = 0
        self._num_tris = 0
        self._num_frames = 0

    def read_header(self, file):
        """
        Read the header from the given file, and stores the data on models.

        Arguments:
        file -- a file object to read from.

        Parsed data is stored in members of the Mdl object because
            a) you can only have a single header per model and
            b) quantities like _num_skins only make sense in terms of an
            actual collection of skins belonging to the model.
            
        Note: private members like _num_verts are only used to store the
        lengths of field parsed from the header of a file we are in the middle
        of reading. Once these collections are read from the binary file, we
        should get the size of collections from len(self.vertices) etc. as per
        the write_header function below.
        
        """
        (self.ident, self.version,
         scale_x, scale_y, scale_z,
         origin_x, origin_y, origin_z,
         self.boundingradius,
         eyeposition_x, eyeposition_y, eyeposition_z,
         self._num_skins,
         self.skinwidth, self.skinheight,
         self._num_verts,
         self._num_tris,
         self._num_frames,
         self.synctype,
         self.flags,
         self.average_size) = _struct_read(Mdl.header_struct, file)
        # run some basic checks that we have been given a model file
        if self.ident != b"IDPO":
            raise ParseError("Ident does not match IDPO")
        if self.version != 6:
            raise ParseError("mdl file is not version 6")
        if self.skinwidth <= 0:
            raise ParseError("mdl file has invalid skinwidth")
        if self.skinheight <= 0:
            raise ParseError("mdl file has invalid skinheight")
        # package the vectors into tuples
        self.scale = (scale_x, scale_y, scale_z)
        self.origin = (origin_x, origin_y, origin_z)
        self.eyeposition = (eyeposition_x, eyeposition_y, eyeposition_z)
        return self

    def write_header(self, file):
        """
        Write a header corresponding to the current Mdl object to a file.
        
        Arguments:
        file -- a file object to write to.

        """
        file.write(Mdl.header_struct.pack
                   (
                       self.ident, self.version,
                       self.scale[0], self.scale[1], self.scale[2],
                       self.origin[0], self.origin[1], self.origin[2],
                       self.boundingradius,
                       self.eyeposition[0],
                       self.eyeposition[1],
                       self.eyeposition[2],
                       len(self.skins),
                       self.skinwidth, self.skinheight,
                       len(self.vertices),
                       len(self.triangles),
                       len(self.frames),
                       self.synctype,
                       self.flags,
                       self.average_size))
        return self

    def recalculate_header(self, reference_frame = 0):
        """
        Calculate updated values for the boundingradius and average_size fields

        This should be called prior to saving for maximum compatibility with
        winquake. reference_frame specifies which model frame to calculate
        the average triangle size from. The default of 0 matches the behaviour
        of the original modelgen tool
        """
        v_x = max(abs(self.origin[0]), abs(self.origin[0]+self.scale[0]))
        v_y = max(abs(self.origin[1]), abs(self.origin[1]+self.scale[1]))
        v_z = max(abs(self.origin[2]), abs(self.origin[2]+self.scale[2]))
        self.boundingradius = math.sqrt(v_x*v_x + v_y*v_y + v_z*v_z)

        def cross(a, b):
            c = [a[1]*b[2] - a[2]*b[1],
                 a[2]*b[0] - a[0]*b[2],
                 a[0]*b[1] - a[1]*b[0]]
            return c

        def area(t):
            p = [self.translate(
                list(self.basic_frames())[reference_frame].vertices[v])
                 for v in t.vertices]
            a = [x-y for x,y in zip(p[0], p[1])]
            b = [x-y for x,y in zip(p[0], p[2])]
            c = cross(a, b)
            return 0.5 * math.sqrt(c[0]*c[0] + c[1]*c[1] + c[2]*c[2])

        self.average_size = sum((area(t) for t in self.triangles))\
                            /len(self.triangles)

    # SKIN

    class Skin:
        """
        Represents a Quake skin.
        
        Pixel data is stored as a flat binary string, each byte is the index
        of the colour of that pixel in the Quake palette.
        
        Object also has height and width of skin stored, but this is not a
        means of controlling the skin size in the model proper. The Mdl class
        stores the actual skin dimensions. These members are solely for
        reference by the methods of the class.
        
        """

        def __init__(self, width, height):
            """
            Create a completely black skin of the desired dimensions

            Arguments:
            width -- the width of this skin
            height -- the height of this skin
            
            """
            self.width = width
            self.height = height
            self.pixels = b"\x00" * width * height

        def read(self, file):
            """
            Read pixel data from the file

            Arguments:
            file -- a file object to read from
            
            """
            self.pixels = file.read(self.width * self.height)
            return self

        def write(self, file, write_header=True):
            """
            Output this skin to a .mdl file.

            Arguments:
            file -- a file object to write to.
            write_header -- boolean indicating whether to write a skin header

            Skingroup objects will contain a list of Skin objects. When writing
            an ungrouped skin we need to write a header indicating the skin is
            ungrouped. Framegroups write their collection of skins by calling this
            method on each skin, but because the Framegroup writes a header for
            the entire skingroup beforehand, it sets the write_header value
            to false.
            
            """
            if write_header:
                file.write(Mdl.single_struct.pack(0))
            file.write(self.pixels)
            return self

        def pixel(self, x, y):
            """
            Return the skin index at the given x,y coordinates.

            Note that the returned value is an index into the quake palette.
            Import qmdl.palette and look the return value up there to get RGB
            values

            """
            x = x % self.width
            y = y % self.height
            return self.pixels[x + y * self.height]

    class SkinGroup:
        """
        Represents a grouping of animated skins
        
        Comprises a header, including a list of durations, followed by a list
        of Skin objects.

        The object stores a width and height value. Changing these will not
        affect the output of the Mdl file in any way, but they will invalidate
        read operations if they are not set correctly.
        
        Unenforced constraint: the list of durations and skins must be equal in
        length or the mdl file will be invalid.

        """

        def __init__(self, width, height):
            """
            Set up a new skingroup with width and height.

            Also creates empty lists for the skins in the group and their
            duration.

            """
            self.width = width
            self.height = height
            self.duration = []
            self.skins = []

        def read(self, file):
            """
            Read a skingroup from the given file

            Arguments:
            file -- a file object to read from

            The first value read determines how many skins are in the group

            """
            count = _struct_read(Mdl.single_struct, file)[0]
            for i in range(count):
                self.duration.append(
                    _struct_read(Mdl.duration_struct, file)[0])
            for i in range(count):
                self.skins.append(Mdl.Skin(self.width, self.height).read(file))
            return self

        def write(self, file):
            """
            Write the skingroup to the given file

            This function writes the header and the durations for the group.
            Then it delegates writing the skins to the Skin objects themselves.
            We use the write_header option to prevent the skins writing another
            header after the group header.

            """
            file.write(Mdl.single_struct.pack(1))
            file.write(Mdl.single_struct.pack(len(self.skins)))
            for dur in self.duration:
                file.write(Mdl.duration_struct.pack(dur))
            for s in self.skins:
                s.write(file, write_header=False)
            return self

    @staticmethod
    def read_list_skins(file, count, width, height):
        """
        Return a list of skins of length count from the given file.

        This is a static method. The function will detect if the next entry in
        the file is a grouped or ungrouped skin and create the appropriate
        entry in the returned list. Callers must be prepared for a list that
        contains Skin and Skingroup objects.

        """
        skin_list = []
        for i in range(count):
            skin_type = _struct_read(Mdl.single_struct, file)[0]
            if skin_type == 0:
                skin = Mdl.Skin(width, height)
            else:
                skin = Mdl.SkinGroup(width, height)
            skin_list.append(skin.read(file))
        return skin_list

    def read_skins(self, file):
        """
        Populate the Mdl skin list with the expected number of skins from the
        given file.

        Note that the list can contain a mixture of Skin and Skingroup objects.
        The expected number counts how many Skin/Skingroup objects there will
        be, a Skingroup with multiple frames still counts 1 towards the total.
        
        """
        self.skins = Mdl.read_list_skins(file,
                                         self._num_skins,
                                         self.skinwidth,
                                         self.skinheight)
        return self

    def write_skins(self, file):
        """
        Write all the skins in the Mdl object to the given file

        Since both Skin and Skingroup implement "write" to correctly record
        themselves to a given file, we don't need to distinguish between the
        two in this method

        """
        for skin in self.skins:
            skin.write(file)
        return self

    def basic_skins(self):
        """
        A generator which iterates over the frames in the model, so that
        a framegroup yields the frames within

        """
        for skin in self.skins:
            try:  # Assume first we have a framegroup
                for basic_skin in skin.skins:
                    yield basic_skin
            except AttributeError:  # just a basic frame
                yield skin

    class Vertex:
        """
        This class stores the skin coordinates for a vertex in the model.

        Skin coordinates in Quake can be "onseam". This is a feature for skins
        which are mapped with a front and back half. Triangles can belong to
        the front or back half of the model. The vertices which are
        "onseam" are the ones attached to both front and back triangle. For
        these vertices, the Vertex records the skin coordinates for the front
        triangles. The coordinate for the back triangles can be calculated by
        adding half the skinwidth to the "u" coordinate.

        The vertex positions in the poses of the model are stored in the Frame
        class. The triangles refer to the vertices by index of their position
        in this list.
        
        Unenforced constraints: The vertex list must remain in sync with the
        triangle and frame list. For example: removing a vertex from this list
        requires you to remove the vertex from every pose in the model frames.
        In addition you must update the triangles list, not just to remove
        every triangle which contained the vertex. You must also to reduce by
        one the index values recorded in each triangle of every vertex which
        follows the deleted vertex, since each will have shifted one place down
        the list.

        """

        def __init__(self):
            """Set up zero defaults for all the class members."""
            self.onseam, self.u, self.v = OFFSEAM_FLAG, 0, 0

        def read(self, file):
            """Read a vertex from the given file."""
            self.onseam, self.u, self.v = _struct_read(Mdl.triple_struct, file)
            return self

        def write(self, file):
            """Write this vertex to the given file."""
            file.write(Mdl.triple_struct.pack(self.onseam, self.u, self.v))
            return self

    def read_vertices(self, file):
        """
        Populate the Mdl vertex list with the expected number of vertices from
        the given file.

        """
        for i in range(self._num_verts):
            self.vertices.append(Mdl.Vertex().read(file))
        return self

    def write_vertices(self, file):
        """
        Write all the vertices from this Mdl object to the given file.

        """
        for vertex in self.vertices:
            vertex.write(file)
        return self

    # TRIANGLE
    # This class represents a triangle in the model as it is stored in the mdl
    # format, a backfacing flag followed by three indices into the vertex list.
    # See VERTEX for the details of the relation between these classes and FRAME
    class Triangle:
        """
        Represents a triangle from a .mdl file.

        The triangle records which vertices it joins as integer offsets to the
        list of vertices on the model. This module does not enforce consistency
        between the vertices and the triangles in this regard. See the Vertex
        class for more details.

        The backface member could have been better named. FACE_BACK = 0, so
        code like
        "if not self.backface"
        is equivalent to
        "if self.backface == FACE_BACK"
        when it appears to carry the opposite meaning. I recommend always using
        the latter form when testing this field.

        """

        def __init__(self):
            """Set up defaults on the members so write cannot fail."""
            self.backface, self.vertices = FACE_FRONT, (0, 0, 0)

        def read(self, file):
            """Read a triangle from the given file."""
            self.backface = _struct_read(Mdl.single_struct, file)[0]
            self.vertices = _struct_read(Mdl.triple_struct, file)
            return self

        def write(self, file):
            """Write this triangle to the given file."""
            file.write(Mdl.single_struct.pack(self.backface) +
                       Mdl.triple_struct.pack(*self.vertices))
            return self

    def coords_for_triangle(self, triangle):
        """
        Return uv coordinates for the triangle

        This method returns the true uv values for each corner of the
        triangle, taking into account backfacing triangles and seam vertices.

        """
        if triangle.backface == FACE_BACK:
            offset = self.skinwidth // 2
        else:
            offset = 0
        raw_coords = [self.vertices[i] for i in triangle.vertices]
        return tuple([(c.u + offset if c.onseam else c.u,
                       c.v) for c in raw_coords])

    def calculate_facings(self):
        """
        Update the back and front facing for triangles.
        
        This should have no effect on a standard ID1 model.
        It is intended for use on full-skinmap conversions which may have
        neglected to set facings. Once run, it may be possible to create seam
        vertices, particularly if the model was originally a ID1 style model.

        Full-skinmap models may have triangles which span both side of the
        skinmap. These triangles are intentionally given FACE_FRONT, which
        ensures that attached vertices can be made onseam without altering their
        position.

        """
        half_way = self.skinwidth // 2
        for triangle in self.triangles:
            tri_coords = self.coords_for_triangle(triangle)
            has_left_vertices = any([coord[0] <= half_way for coord in tri_coords])
            if not has_left_vertices:
                triangle.backface = FACE_BACK
            else:
                triangle.backface = FACE_FRONT
        return self

    def read_triangles(self, file):
        """
        Populate the Mdl triangle list with the expected number of triangles
        from the given file.

        """
        for i in range(self._num_tris):
            self.triangles.append(Mdl.Triangle().read(file))
        return self

    def write_triangles(self, file):

        """
        Write all the triangles from this Mdl object to the given file.

        """
        for triangle in self.triangles:
            triangle.write(file)
        return self

    class Coord:
        """
        This class stores the 'compressed' coordinate the mdl file uses for
        vertex position.

        x, y, z values are stored as a byte in the range 0-255. Vertex normals
        are stored in a 4th byte in the range 0-161. The normal value can be
        looked up in the standard table of normals provided in anorms.h
        
        This class contains a pair of helper functions to deal with the normals
        'decode' returns the normal vector that the coord stores.
        'encode' takes a vector and sets the normal of the coord to the index
        of the standard normal which best approximates its direction.
        See also the Mdl method called 'translate'.

        """

        def __init__(self):
            """Initialise all the class members with valid values."""
            self.position, self.normal = (0, 0, 0), 0

        def read(self, file):
            """Read position and normal from the given file."""
            coord = _struct_read(Mdl.coord_struct, file)
            self.position = coord[0:3]
            self.normal = coord[3]
            return self

        def write(self, file):
            """Write the position and normal of this coordinate to the given file"""
            file.write(Mdl.coord_struct.pack(*(self.position + (self.normal,))))
            return self

        def encode(self, vector):
            """
            Set the normal value of this coordinate to approximate the given
            normal.

            Quake has a set table of normals, and we store an index to a table
            entry. We calculate the dot product between each normal and the
            given vector. The normal which produces the highest dot product is
            set

            """
            m = 0
            n = 0
            # To speed up this time-consuming inner loop, we have precomputed
            # 8 tables of vertices to consider depending on the quadrant
            # that the vertex lies in
            quadrant = quadrants[(math.copysign(1, vector[0]),
                                  math.copysign(1, vector[1]),
                                  math.copysign(1, vector[2]))]
            for i in quadrant:
                dot = vector[0] * normals[i][0] + vector[1] * normals[i][1] + \
                    vector[2] * normals[i][2]
                if dot > m:
                    m = dot
                    n = i
            self.normal = n
            return self

        def decode(self):
            """
            Look up this co-ordinate's normal value and returns the entry.

            This returns a tuple of 3 floats, giving a vector of length 1 which
            points along the normal at this coordinate.            

            """
            return normals[self.normal]

    def translate(self, coord):
        """
        Return the world coordinates for the given Coord object.

        The return is a 3-tuple of floats. The scaling of the compressed
        coordinates is performed according to the members of the current Mdl
        object, as they can't be interpreted independent of a model

        """
        return tuple(org + scale * x for org, scale, x in zip(
            self.origin, self.scale, coord.position))

    class Frame:
        """
        Represents a single animation frame in a .mdl file

        This class is used to represent both standalone frames and the
        individual frames in a framegroup.

        To create a valid .mdl file, it is required that each frame has the
        correct number of vertices. This constraint is unenforced, take care if
        you are creating or deleting vertices in your script.

        """

        def __init__(self):
            """
            Initialise all the members

            """
            self.bbox_min = Mdl.Coord()
            self.bbox_max = Mdl.Coord()
            self.name = b"nameless"
            self.vertices = []

        def read(self, file, vertcount):
            """
            Read a frame from the given file

            Arguments:
            file -- a file object to read from.
            vertcount -- the number of vertices to read into the frame

            vertcount should normally be set according to the vertex count read
            in the model file's header            

            """
            self.bbox_min.read(file)
            self.bbox_max.read(file)
            self.name = file.read(16).split(b"\x00")[0]
            # Loading vertices from frames is the inner loop of the reading
            # process. So we slightly break the encapsulation here in order
            # to halve the time taken to read the file            
            vert_data = file.read(4 * vertcount)
            for i in range(vertcount):
                c = Mdl.Coord()
                c.position = tuple(vert_data[(i * 4):(i * 4 + 3)])
                c.normal = vert_data[i * 4 + 3]
                self.vertices.append(c)
            return self

        def write(self, file, write_header=True):
            """
            Write the content of this frame to the given file.

            When this Frame is representing one of the poses in a Framegroup,
            the Framegroup is responsible for writing one header for all the
            frames before they write themselves. In this case it is necessary
            to skip the standard frame header, hence the write_header parameter.

            """
            if write_header:
                file.write(Mdl.single_struct.pack(0))
            self.bbox_min.write(file)
            self.bbox_max.write(file)
            file.write(Mdl.name_struct.pack(self.name))
            for vert in self.vertices:
                vert.write(file)
            return self

        def calculate_bounds(self):
            """
            Update the mins and max to reflect the current coordinates for
            vertices in this frame.

            This is called prior to writing the model to disk to ensure
            compatibility with software quake (GL engines tend to survive even
            where incorrect values are set here)
            """
            c_x, c_y, c_z = zip(*(c.position for c in self.vertices))
            self.bbox_min.position = (min(c_x),min(c_y),min(c_z))
            self.bbox_max.position = (max(c_x),max(c_y),max(c_z))

    class FrameGroup:
        """
        Stores a group of frames to be animated automatically.

        The frames are all stored as conventional Frame objects.
        
        The duration field is misnamed, as it records the amount of time after
        a loop begins that the corresponding frame should end. The
        misinterpretation extends to most Quake engines after Winquake, who
        usually read just the first value in the duration array and animate the
        whole sequence at that rate.

        The sequencemin and sequencemax values are initially read from the
        file. There is a helper method called calculate_bounds which will
        update these values to reflect the object's current set of frames, and
        this function is run each time the object is written to a file.
        
        """

        def __init__(self):
            """
            Set up all the members of the class, in particular the arrays.

            """
            self.sequencemin = Mdl.Coord()
            self.sequencemax = Mdl.Coord()
            self.duration = []
            self.frames = []

        def read(self, file, vertcount):
            """
            Read a framegroup from the given file
            
            Arguments:
            file -- a file object to read from.
            vertcount -- the number of vertices to read into the frame
            
            The vertex count has to be passed to the function to determine the
            size of the frame entries in the file.

            """
            count = _struct_read(Mdl.single_struct, file)[0]
            self.sequencemin.read(file)
            self.sequencemax.read(file)
            for i in range(count):
                self.duration.append(_struct_read(Mdl.duration_struct, file)[0])
            for i in range(count):
                self.frames.append(Mdl.Frame().read(file, vertcount))
            return self

        def calculate_bounds(self):
            """
            Recalculate the sequencemin and sequencemax members.

            This function is always called ahead of the object writing itself
            to disk to ensure written mdl files are consistent.

            """
            _sequencemin = (255, 255, 255)
            _sequencemax = (0, 0, 0)
            for frame in self.frames:
                frame.calculate_bounds()
                _bbox_min = frame.bbox_min.position
                _bbox_max = frame.bbox_max.position
                _sequencemin = (min(_sequencemin[0], _bbox_min[0]),
                                min(_sequencemin[1], _bbox_min[1]),
                                min(_sequencemin[2], _bbox_min[2]))
                _sequencemax = (max(_sequencemax[0], _bbox_max[0]),
                                max(_sequencemax[1], _bbox_max[1]),
                                max(_sequencemax[2], _bbox_max[2]))
            self.sequencemin.position = _sequencemin
            self.sequencemax.position = _sequencemax

        def write(self, file):
            """
            Write this framegroup to the given file.

            This function depends on the write method on the Frame object, see
            that method for details on how the header is handled.

            """
            file.write(Mdl.single_struct.pack(1))
            file.write(Mdl.single_struct.pack(len(self.frames)))
            self.sequencemin.write(file)
            self.sequencemax.write(file)
            for dur in self.duration:
                file.write(Mdl.duration_struct.pack(dur))
            for frame in self.frames:
                frame.write(file, write_header=False)
            return self


    @staticmethod
    def read_list_frame(file, frame_count, vert_count):
        """
        Return a list of frame_count frames or framegroups from a file.

        vert_count is needed to determine how long each frame entry is. The
        frame_count determines the length of the list returned, and so each
        framegroup counts one towards the length of the list

        """
        frame_list = []
        for i in range(frame_count):
            frame_type = _struct_read(Mdl.single_struct, file)[0]
            if frame_type == 0:
                frame = Mdl.Frame()
            else:
                frame = Mdl.FrameGroup()
            frame_list.append(frame.read(file, vert_count))
        return frame_list

    def read_frames(self, file):
        """
        Read the expected number of frames or framegroups from the given file.

        Note that the number of frames in the header of the .mdl file counts
        each framegroup as just one frame, regardless of the size of the group.

        """
        self.frames = Mdl.read_list_frame(file,
                                          self._num_frames,
                                          self._num_verts)
        return self

    def write_frames(self, file):
        """
        Write all the frames in the Mdl to a file.

        """
        for frame in self.frames:
            frame.write(file)
        return self

    def basic_frames(self):
        """
        A generator which iterates over the frames in the model, so that
        a framegroup yields the frames within

        """
        for frame in self.frames:
            try:  # Assume first we have a framegroup
                for basic_frame in frame.frames:
                    yield basic_frame
            except AttributeError:  # just a basic frame
                yield frame

    def read(self, file):
        """
        Read the entire contents of a .mdl file into this Mdl object.

        """
        self.read_header(file)
        self.read_skins(file)
        self.read_vertices(file)
        self.read_triangles(file)
        self.read_frames(file)
        return self

    def write(self, file):
        """
        Write the data from this Mdl object to file in .mdl format.

        """
        self.write_header(file)
        self.write_skins(file)
        self.write_vertices(file)
        self.write_triangles(file)
        self.write_frames(file)
        return self
